<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * @psalm-type Payload array{
 *   priceList: string,
 *   reason: string,
 * }
 */
readonly class ProductPricingPricelistChanged implements EventInterface
{
    public static function getMessageType(): string
    {
        return "AwarditPricingPricelistChangedV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-pricing-pricelist-changed";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        return new self(
            priceList: $message->getEventEntityId(),
            reason: $payload["reason"],
            revision: $message->getEventRevision(),
            archived: $message->getEventIsDeleted(),
        );
    }

    public function __construct(
        /**
         * Human-readable price list code for the price list which this
         * product-price is part of.
         *
         * Examples:
         *   * "sas_sek"
         *   * "sgdsextra_nok"
         */
        public string $priceList,
        /**
         * Update reason
         */
        public string $reason,
        /**
         * Product price revision for the particular product in the price list.
         */
        public int $revision,
        /**
         * If this price-list entry is archived.
         */
        public bool $archived,
    ) {
        assert($revision > 0);
    }

    public function getEntityId(): string
    {
        return $this->priceList;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    public function getRevision(): int
    {
        return $this->revision;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "priceList" => $this->priceList,
            "reason" => $this->reason,
        ];
    }
}
