<?php

class Awardit_ProductEmail_Model_Observer extends Mage_Core_Model_Abstract {

    private const TRIGGER_ON = ['processing', 'complete'];

    /**
     * Observer listening to order change to possibly send emails.
     */
    public function sendProductEmails(Varien_Event_Observer $observer): void {
        /** @var Mage_Sales_Model_Order */
        $order = $observer->getOrder();

        if (!in_array($order->getStatus(), self::TRIGGER_ON)) {
            return; // Current status should not trigger email
        }
        foreach ($order->getAllStatusHistory() as $historic_status) {
            if (substr_count($historic_status->getComment(), 'Sent product specific email') > 0) {
                return; // Has already sent email
            }
        }

        // Load SKU/template map
        $map = $this->getSkuTemplateMap();
        if (empty($map)) {
            return; // Nothing to do here
        }

        // Load available email templates
        $templates = $this->getTemplates();
        if (empty($templates)) {
            return; // Nothing to do here
        }

        // Traverse products, and email tempalte on matching SKU:s
        foreach ($order->getAllVisibleItems() as $item) {
            if ($item->getEmailSent()) {
                continue;
            }

            // Init mail queue
            $queue = Mage::getModel('core/email_queue');
            $queue->setEntityId($item->getId())
                ->setEntityType("item")
                ->setEventType('product_confirmation');

            $product = $item->getProduct();
            $sku = $product->getSku();
            if (!array_key_exists($sku, $map)) {
                continue; // Product not mapped to template
            }
            $template_id = $map[$sku];
            if (!array_key_exists($template_id, $templates)) {
                continue; // Template do not exist (might have been removed)
            }
            $template = $templates[$template_id];
            $template->setQueue($queue);

            $template->sendTransactional(
                $template_id,
                'general',
                $order->getCustomerEmail(),
                Mage::helper('customer')->getFullCustomerName($order),
                [
                    'customer'        => Mage::helper('customer')->getFullCustomerName($order),
                    'customerEmail'   => $order->getCustomerEmail(),
                    'billingAddress'  => $order->getBillingAddress(),
                    'shippingAddress' => $order->getShippingAddress(),
                    'product'         => $product,
                    'item'            => $item,
                    'order'           => $order,
                ]
            );
            $comment = "Sent product specific email for '{$sku}' using '{$template->getTemplateCode()}'.";
            $order->addStatusToHistory($order->getStatus(), $comment, true)
                ->save();
            $item->setEmailSent(true);
        }
    }

    private function getTemplates(): array {
        $templates = [];
        $collection = Mage::getModel('core/email_template')->getCollection() ?: [];
        foreach ($collection as $template) {
            $templates[$template->getId()] = $template;
        }
        return $templates;
    }

    private function getSkuTemplateMap(): array {
        $unserializer = Mage::helper("core/unserializeArray");
        /** @var string */
        $map = Mage::getStoreConfig('awardit_productemail/awardit_productemail_confirmation/confirmation_template');
        if (empty($map)) {
            return []; // Might be null
        }
        $map = $unserializer->unserialize($map);
        /** @var array<string, string> */
        $return = [];
        foreach ($map as $item) {
            if (empty($item['sku']) || empty($item['email_template_id'])) {
                continue; // Not a valid connection
            }
            $return[$item['sku']] = $item['email_template_id'];
        }
        return $return;
    }
}
