<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

/**
 * Test observer adding email to queue
 */
class ObserverTest extends TestCase {

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();
        throw $e;
    }

    public function testObserver(): void {
        MagentoManager::init('default');

        $content = implode(", ", [
            "customer: {{var customer}}",
            "email: {{var customerEmail}}",
            "country: {{var billingAddress.country_id}}",
            "qty: {{var item.qty}}",
            "sku: {{var product.sku}}",
            "product: {{var product.name}}",
            "currency: {{var order.order_currency_code}}",
        ]);

        // Prepare some test data
        $this->addSender();
        $template = $this->addTemplate($content);
        $product = $this->addProduct();
        $this->mapTemplateToProduct($template, $product);
        $order = $this->createOrder([$product]);

        $content_target = implode(", ", [
            "customer: Test Testmann",
            "email: test@awardit.com",
            "country: DE",
            "qty: 2",
            "sku: {$product->getSku()}",
            "product: {$product->getName()}",
            "currency: SEK",
        ]);

        // Call observer
        $ev = new Varien_Event_Observer();
        $ev->setOrder($order);
        $observer = Mage::getModel("awardit_productemail/observer");
        $observer->salesOrderPlaceAfter($ev);

        // Verify added to mail queue
        $collection = Mage::getModel('core/email_queue')->getCollection();
        $collection->getSelect()->where(
            "event_type='product_confirmation' AND message_body LIKE '%{$product->getSku()}%'
        ");

        $this->assertCount(1, $collection);
        foreach ($collection as $qmail) {
            $this->assertEquals($content_target, $qmail->getMessageBody());
            $recipients = $qmail->getRecipients();
            $this->assertEquals('test@awardit.com', $recipients[0][0]);
            $this->assertEquals('Test Testmann', $recipients[0][1]);
        }
    }

    private function addSender(): void {
        Mage::getConfig()->saveConfig('trans_email/ident_general/name', 'Test sender');
        Mage::getConfig()->saveConfig('trans_email/ident_general/email', 'test@awardit.com');
        Mage::getConfig()->cleanCache();
    }

    private function addTemplate(string $content): Mage_Core_Model_Email_Template {
        $key = rand(1000, 9999);
        $template = Mage::getModel('core/email_template');
        $template->setTemplateCode("Template name {$key}");
        $template->setTemplateSubject("Template subject {$key}");
        $template->setTemplateText($content);
        $template->save();
        return $template;
    }

    private function addProduct(): Mage_Catalog_Model_Product {
        $key = rand(1000, 9999);
        $product = Mage::getModel('catalog/product');
        $product->setSku("SKU {$key}");
        $product->setName("Test product {$key}");
        $product->setUrlKey("test-{$key}");
        $product->setAttributeSetId($product->getDefaultAttributeSetId());
        $product->save();
        return $product;
    }

    private function mapTemplateToProduct(
        Mage_Core_Model_Email_Template $template,
        Mage_Catalog_Model_Product $product
    ): void {
        $key = rand(1000, 9999);
        $map = ["_{$key}" => ['sku' => $product->getSku(), 'email_template_id' => $template->getId()]];
        $map = serialize($map);
        Mage::getConfig()->saveConfig(
            "awardit_productemail/awardit_productemail_confirmation/confirmation_template",
            $map,
            "default",
            0
        );
        Mage::getConfig()->cleanCache();
    }

    private function createOrder(array $products): Mage_Sales_Model_Order {
        $order = Mage::getModel('sales/order');
        $order->save();
        foreach ($products as $product) {
            $item = Mage::getModel('sales/order_item');
            $item->setOrder($order);
            $item->setProduct($product);
            $item->setQty(2);
            $order->addItem($item);
        }
        $order->setOrderCurrencyCode('SEK');
        $order->setCustomerEmail('test@awardit.com');
        $order->setCustomerFirstname('Test');
        $order->setCustomerLastname('Testmann');
        $order->setBillingAddress(Mage::getModel("sales/order_address")->addData([
            "country_id" => "DE",
        ]));
        $order->setShippingAddress(Mage::getModel("sales/order_address")->addData([
            "country_id" => "DE",
        ]));

        return $order;
    }
}
