<?php

class Awardit_ProductEmail_Model_Observer extends Mage_Core_Model_Abstract
{
    private const TRIGGER_ON = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
        Mage_Sales_Model_Order::STATE_COMPLETE,
    ];

    /**
     * Observer listening to order change to possibly send emails.
     */
    public function sendProductEmails(Varien_Event_Observer $observer): void
    {
        /** @var Mage_Sales_Model_Order */
        $order = $observer->getOrder();
        /** @var Awardit_ProductEmail_Helper_Data */
        $helper = Mage::helper('awardit_productemail');

        $helper->log("Observer triggered: OldStatus = {$order->getOrigData("status")}, NewStatus = '{$order->getStatus()}', oldState = {$order->getOrigData("state")}, newState = '{$order->getState()}'");

        // @todo: Should check state, but serial code payments sets incorrect state as of now.
        if (!in_array($order->getStatus(), self::TRIGGER_ON)) {
            $helper->log("Wrong newStatus, bail out");
            return; // Current status should not trigger email
        }
        if (in_array($order->getOrigData('status'), self::TRIGGER_ON)) {
            $helper->log("Wrong oldStatus, bail out");
            return; // Previous status should not trigger email
        }

        // Load SKU/template map
        $map = $this->getSkuTemplateMap();
        if (empty($map)) {
            $helper->log("Found no trigger item, bail out");
            return; // Nothing to do here
        }

        // Load available email templates
        $templates = $this->getTemplates();
        if (empty($templates)) {
            $helper->log("Found no email template, bail out");
            return; // Nothing to do here
        }

        // Traverse products, and email template on matching SKU:s
        foreach ($order->getAllVisibleItems() as $item) {
            $additional = $item->getAdditionalData() ? (unserialize($item->getAdditionalData(), ["allowed_classes" => false]) ?: [ "product_emails" => [] ]) : [ "product_emails" => [] ];

            if (!empty($additional["product_emails"])) {
                $helper->log("Already sent email for this item [{$item->getSku()}], trying next");
                continue; // Already sent for this item
            }

            // Get product
            $product = $this->getProduct($item, $map);
            if (!$product) {
                $helper->log("Product [{$item->getSku()}] not mapped to template, trying next");
                continue; // Product not mapped to template
            }
            $sku = $product->getSku();

            // Init mail queue
            $queue = Mage::getModel('core/email_queue');
            $queue->setEntityId((int)$item->getId())
                ->setEntityType('item')
                ->setEventType('product_confirmation');

            // Traverse connected templates
            foreach ($map[$sku] as $template_id) {
                if (!array_key_exists($template_id, $templates)) {
                    continue; // Template do not exist (might have been removed)
                }
                if (in_array($template_id, $additional['product_emails'])) {
                    continue; // Already sent using this template
                }

                $template = $templates[$template_id];
                $template->setQueue($queue);

                $template->sendTransactional(
                    $template_id,
                    'general',
                    $order->getCustomerEmail(),
                    Mage::helper('customer')->getFullCustomerName($order),
                    [
                        'customer'        => Mage::helper('customer')->getFullCustomerName($order),
                        'customerEmail'   => $order->getCustomerEmail(),
                        'billingAddress'  => $order->getBillingAddress(),
                        'shippingAddress' => $order->getShippingAddress(),
                        'product'         => $product,
                        'item'            => $item,
                        'order'           => $order,
                    ]
                );

                $helper->log("Email sent to {$order->getCustomerEmail()}");
                $comment = "Sent product specific email for \"{$sku}\" using \"{$template->getTemplateCode()}\".";
                $additional['product_emails'][] = $template_id;
                $history = $order
                    ->addStatusHistoryComment($comment)
                    ->setIsCustomerNotified(true);
                $history->save();
            }

            // Save sent emails
            if (!empty($additional)) {
                $item->setAdditionalData(serialize($additional));
                $item->save();
            }
        }
    }

    private function getProduct(Mage_Sales_Model_Order_Item $item, array $map): ?Mage_Catalog_Model_Product
    {
        $product = $item->getProduct();
        if (array_key_exists($product->getSku(), $map)) {
            return $product; // Direct match
        }
        foreach ($item->getChildrenItems() as $child) {
            if ($match = $this->getProduct($child, $map)) {
                return $match; // Match on child
            }
        }
        return null;
    }

    private function getTemplates(): array
    {
        $templates = [];
        $collection = Mage::getModel('core/email_template')->getCollection() ?: [];
        foreach ($collection as $template) {
            $templates[$template->getId()] = $template;
        }
        return $templates;
    }

    private function getSkuTemplateMap(): array
    {
        $unserializer = Mage::helper("core/unserializeArray");
        /** @var string */
        $map = Mage::getStoreConfig('awardit_productemail/awardit_productemail_confirmation/confirmation_template');
        if (empty($map)) {
            return []; // Might be null
        }
        $map = $unserializer->unserialize($map);
        /** @var array<string, string> */
        $return = [];
        foreach ($map as $item) {
            if (empty($item['sku']) || empty($item['email_template_id'])) {
                continue; // Not a valid connection
            }
            $return[$item['sku']][] = $item['email_template_id'];
        }
        return $return;
    }
}
