<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

/**
 * Test observer adding email to queue
 */
class ObserverTest extends TestCase {

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();
        throw $e;
    }

    public function testObserver(): void {
        MagentoManager::init('default');

        // Prepare some test data
        $this->addSender();
        $template = $this->addTemplate();
        $product = $this->addProduct();
        $this->mapTemplateToProduct($template, $product);
        $order = $this->createOrder([$product]);

        // Call observer
        $ev = new Varien_Event_Observer();
        $ev->setOrder($order);
        $observer = Mage::getModel("awardit_productemail/observer");
        $observer->salesOrderPlaceAfter($ev);

        // Verify added to mail queue
        $collection = Mage::getModel('core/email_queue')->getCollection();
        $collection->getSelect()->where(
            "event_type='product_confirmation' AND message_body='{$template->getTemplateText()}'
        ");

        $this->assertCount(1, $collection);
    }

    private function addSender(): void {
        Mage::getConfig()->saveConfig('trans_email/ident_general/name', 'Test sender');
        Mage::getConfig()->saveConfig('trans_email/ident_general/email', 'test@awardit.com');
        Mage::getConfig()->cleanCache();
    }

    private function addTemplate(): Mage_Core_Model_Email_Template {
        $key = rand(1000, 9999);
        $template = Mage::getModel('core/email_template');
        $template->setTemplateCode("Template name {$key}");
        $template->setTemplateSubject("Template subject {$key}");
        $template->setTemplateText("Template content {$key}");
        $template->save();
        return $template;
    }

    private function addProduct(): Mage_Catalog_Model_Product {
        $key = rand(1000, 9999);
        $product = Mage::getModel('catalog/product');
        $product->setSku("SKU {$key}");
        $product->setUrlKey("test-{$key}");
        $product->setAttributeSetId($product->getDefaultAttributeSetId());
        $product->save();
        return $product;        
    }

    private function mapTemplateToProduct(
        Mage_Core_Model_Email_Template $template, 
        Mage_Catalog_Model_Product $product
    ): void {
        $key = rand(1000, 9999);
        $map = ["_{$key}" => ['sku' => $product->getSku(), 'email_template_id' => $template->getId()]];
        $map = serialize($map);
        Mage::getConfig()->saveConfig(
            "awardit_productemail/awardit_productemail_confirmation/confirmation_template", 
            $map, 
            "default", 
            0
        );
        Mage::getConfig()->cleanCache();
    }

    private function createOrder(array $products): Mage_Sales_Model_Order {
        $order = Mage::getModel('sales/order');
        $order->save();
        foreach ($products as $product) {
            $item = Mage::getModel('sales/order_item');
            $item->setOrder($order);
            $item->setProduct($product);
            $order->addItem($item);
        }
        return $order;
    }

}
