<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GraphQL\Utils\SchemaPrinter;
use MageQL\Context;
use PHPUnit\Framework\Assert;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\Driver;
use Spatie\Snapshots\MatchesSnapshots;

/**
 * Test pop-up component through GraphQL.
 */
class PopupTest extends TestCase implements Driver {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    // We also implement a test-driver to ensure we use the graphql extension
    public function serialize($data): string {
        return $data;
    }

    public function extension(): string {
        return "graphql";
    }

    public function match($expected, $actual): void {
        Assert::assertEquals($expected, $this->serialize($actual));
    }

    // Schema test
    public function testSchema(): void {

        MagentoManager::init('default');

        $app = Mage::app();
        $store = $app->getStore();
        $helper = Mage::helper("mageql/data");
        $layout = Mage::getSingleton("core/layout");

        $layout->setArea(MageQL_Core_Router_GraphQL::GRAPHQL_AREA);
        $app->loadArea($layout->getArea());

        $session = Mage::getSingleton("core/session", ["name" => "frontend"])->start();
        $context = new Context($store, $session, "default");

        // Test schema
        $schema = $helper->loadSchema($context, [
            "unreachable" => true,
        ]);
        $sdl = SchemaPrinter::doPrint($schema, [
            "schemaDescriptions" => true,
        ]);
        $this->assertStringContainsString('StoreInfo', $sdl);
        $this->assertStringContainsString('PopUp', $sdl);
        $this->assertMatchesSnapshot($sdl, $this);
    }

    public function testDisabledPopup(): void {

        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig("awardit_popupbanner/popup_banner/enabled", "0", "default", 0);
        $config->saveConfig("awardit_popupbanner/popup_banner/body", "Should be disabled", "default", 0);

        // Test response
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {info{popUp{body}}}'), 'default');

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
 
     public function testEnabledPopup(): void {

        MagentoManager::init('default');

        $config = new Mage_Core_Model_Config();
        $config->saveConfig("awardit_popupbanner/popup_banner/enabled", "1", "default", 0);
        $config->saveConfig("awardit_popupbanner/popup_banner/body", "Some <b>html</b> content", "default", 0);

        // Test response
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {info{popUp{body}}}'), 'default');

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }  
}
