<?php

declare(strict_types=1);

use GuzzleHttp\Cookie\CookieJar;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\ClientException;
use Psr\Http\Message\ResponseInterface;
use GuzzleHttp\Psr7\Uri;

/**
 * @psalm-type ApiPointsData array{
 *   activepoints: string,
 *   exactpoints: string,
 *   rankpoints: string,
 *   pendingpoints: string,
 *   grouppoints: string,
 *   exactgrouppoints: string,
 *   grouprankpoints: string,
 *   ownpoints: string,
 *   exactownpoints: string,
 *   ownrankpoints: string,
 *   expirepoints: string,
 *   expiredate: string
 * }
 * @psalm-type CheckoutData object{
 *   pointSources: array{object{
 *     name: string,
 *     vismaId: string,
 *     items: array{object{
 *       sku: string,
 *       productId: string,
 *       productName: string,
 *       amount: float,
 *       vat: float
 *     }}
 *   }}
 * }
 * @psalm-type JavaSession array{Cookie_2fa:string, JSESSIONID:string, partneruserid:?int, memberuserid:string}
 * @psalm-type EmailVerifyType "requestVerification"|"verificationEmailSent"|"twofactor"
 * @psalm-type EmailVerify array{
 *   emailverify:EmailVerifyType,
 *   JSESSIONID:?string,
 *   partneruserid:?int,
 *   memberuserid:string,
 *   mailto:?string,
 *   twofactorSkipDays:?int
 * }
 * @psalm-type JavaError array{
 *   code:int,
 *   message:string,
 *   tf_tryleft:?int,
 *   tf_tryagain:?bool,
 *   tf_wait:?int,
 *   tf_type:?int,
 *   tf_type_str:?string
 * }
 * @psalm-type AccountConfig "required"|"readonly"|"optional"|"hidden"
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_Points_Model_Api extends Mage_Core_Model_Abstract
{
    public function filterPoints(string $pointsData): int
    {
        return (int)preg_replace("/[^-0-9]/", "", $pointsData);
    }

    protected function logRequest(
        Mage_Core_Model_Store $store,
        string $method,
        string $url,
        array $options,
        ?ResponseInterface $res,
        float $time
    ): void {
        $helper = Mage::helper("awardit_points");
        $json = $options["json"] ?? null;

        Mage::log(sprintf(
            "AwarditAPI: %s %s%s%s, response: %d %s, took %f ms",
            $method,
            $helper->getServer($store),
            $url,
            $json ? ": " . json_encode($json) : "",
            $res ? $res->getStatusCode() : 0,
            $res ? (string)$res->getBody() : "<empty>",
            $time * 1000
        ), Zend_Log::DEBUG, "awardit_points_api");
    }

    /**
     * @param bool $isTest
     */
    protected function createClient(Mage_Core_Model_Store $store): Client
    {
        $helper = Mage::helper("awardit_points");
        $url = $helper->getServer($store);
        $request = Mage::app()->getRequest();
        $remoteAddr = $request->getServer("REMOTE_ADDR");
        $realIp = $request->getServer("HTTP_X_REAL_IP");
        $userAgent = $request->getServer("HTTP_USER_AGENT");
        $xForwardedFor = $request->getServer("HTTP_X_FORWARDED_FOR")
            ? explode(", ", $request->getServer("HTTP_X_FORWARDED_FOR"))
            : [];
        $xForwardedFor[] = $remoteAddr;

        return new Client([
            "base_uri" => $url,
            "debug"    => false,
            "headers"  => [
                "Accept" => "application/json",
                "Authorization" => $helper->getApiKey($store),
                "User-Agent" => $userAgent,
                "X-Forwarded-For" => implode(", ", $xForwardedFor),
                "X-Real-IP" => $realIp,
            ],
        ]);
    }

    protected function request(
        Mage_Core_Model_Store $store,
        string $method,
        string $url,
        array $options
    ): ResponseInterface {
        $client = $this->createClient($store);
        $start = microtime(true);

        try {
            $res = $client->request($method, $url, $options);
            $end = microtime(true);

            $this->logRequest($store, $method, $url, $options, $res, $end - $start);

            return $res;
        } catch (Exception $e) {
            $end = microtime(true);

            $this->logRequest(
                $store,
                $method,
                $url,
                $options,
                $e instanceof BadResponseException ? $e->getResponse() : null,
                $end - $start
            );

            throw $e;
        }
    }

    /**
     * @return mixed
     */
    public function decodeJson(string $bytes)
    {
        $encoding = mb_detect_encoding($bytes, "UTF-8,ISO-8859-1,WINDOWS-1252", true);
        $body = $encoding && $encoding !== "UTF-8" ? mb_convert_encoding($bytes, "UTF-8", $encoding) : $bytes;

        $val = json_decode($body, true);

        // simulate JSON_THROW_ON_ERROR for PHP <7.3
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception(json_last_error_msg());
        }

        return $val;
    }

    /**
     * @param Mage_Core_Model_Store $store
     * @param string $username,
     * @param string $password
     * @return ?JavaSession|EmailVerify|JavaError
     */
    public function loginCustomer(
        Mage_Core_Model_Store $store,
        string $username,
        string $password
    ): ?array {
        $helper = Mage::helper("awardit_points");
        $client = $this->createClient($store);
        $host = $helper->getServer($store);
        $url = sprintf("partners/%s/login", $helper->getPartnerId($store));
        $body = ["memberid" => $username, "password" => $password];
        return $this->_login($client, $host, $url, $body);
    }

    /**
     * @param Mage_Core_Model_Store $store
     * @param string $code
     * @param bool $remember
     * @return ?JavaSession|EmailVerify|JavaError
     */
    public function loginCustomerOtp(
        Mage_Core_Model_Store $store,
        string $code,
        bool $remember
    ): ?array {
        $helper = Mage::helper("awardit_points");
        $client = $this->createClient($store);
        $host = $helper->getServer($store);
        $url = sprintf("partners/%s/logintwo", $helper->getPartnerId($store));
        $body = ["code" => $code, "remember" => $remember];
        return $this->_login($client, $host, $url, $body);
    }

    /**
     * @param Mage_Core_Model_Store $store
     * @param string $loginkey,
     * @param string $hiddenlogin
     * @return ?JavaSession|EmailVerify|JavaError
     */
    public function loginAsCustomer(
        Mage_Core_Model_Store $store,
        string $loginkey,
        string $hiddenlogin
    ): ?array {
        $helper = Mage::helper("awardit_points");
        $client = $this->createClient($store);
        $host = $helper->getServer($store);
        $url = sprintf("partners/%s/login", $helper->getPartnerId($store));
        $body = ["loginkey" => $loginkey, "hiddenlogin" => $hiddenlogin];
        return $this->_login($client, $host, $url, $body);
    }

    /**
     * @param Client $client
     * @param string $host
     * @param string $url
     * @param array{
     *   loginkey:string,
     *   hiddenlogin:string
     * }|array{
     *   username:string,
     *   password:string
     * }|array{
     *   code:string,
     *   remember: bool
     * } $body
     * @return ?JavaSession|EmailVerify|JavaError
     */
    private function _login(
        Client $client,
        string $host,
        string $url,
        array $body
    ): ?array {
        $helper = Mage::helper("awardit_points");
        $clientCookies = $helper->getRequestCookies();

        $cookieHost = (new Uri($host))->getHost();
        $jar = CookieJar::fromArray($clientCookies, $cookieHost);

        try {
            $res = $client->request("POST", $url, [
                "cookies" => $jar,
                "json" => $body,
            ]);

            if ($res->getStatusCode() !== 200) {
                return null;
            }

            /**
             * @var false|array{
             *   login:int,
             *   partneruserid?:?int,
             *   emailverify?:EmailVerifyType,
             *   memberuserid:string,
             *   tf_save?:int
             * }
             */
            $data = $this->decodeJson((string)$res->getBody());

            if (! is_array($data) || ! array_key_exists("login", $data)) {
                Mage::log(
                    sprintf(
                        "AwarditAPI: Bad 200 response from login, POST %s%s (%s):  %s",
                        $host,
                        $url,
                        json_encode(array_intersect_key($body, ["loginkey" => true, "memberid" => true])),
                        (string)$res->getBody()
                    ),
                    Zend_Log::DEBUG,
                    "awardit_points_api"
                );

                return null;
            }

            Mage::log(sprintf(
                "AwarditAPI: POST %s%s (%s), response: %d %s",
                $host,
                $url,
                json_encode(array_intersect_key($body, ["loginkey" => true, "memberid" => true])),
                $res->getStatusCode(),
                (string)$res->getBody(),
            ), Zend_Log::DEBUG, "awardit_points_api");

            $sessionCookie = $jar->getCookieByName("JSESSIONID");
            $deviceCookie = $jar->getCookieByName("Cookie_2fa");

            if (! empty($data["emailverify"])) {
                /** @var EmailVerify */
                return [
                    "emailverify" => $data["emailverify"],
                    "JSESSIONID" => $sessionCookie ? $sessionCookie->getValue() : null,
                    "partneruserid" => empty($data["partneruserid"]) ? null : (int)$data["partneruserid"],
                    "memberuserid" => empty($data["memberuserid"]) ? null : $data["memberuserid"],
                    "mailto" => empty($data["mailto"]) ? null : $data["mailto"],
                    "twofactorSkipDays" => empty($data["tf_save"]) ? 0 : $data["tf_save"],
                ];
            }

            if (empty($data["login"])) {
                return null;
            }

            if (! $sessionCookie) {
                throw new Exception(sprintf("%s: Missing cookie from login", __METHOD__));
            }

            /** @var JavaSession */
            return [
                "JSESSIONID" => $sessionCookie->getValue(),
                "Cookie_2fa" => $deviceCookie ? $deviceCookie->getValue() : null,
                "partneruserid" => empty($data["partneruserid"]) ? null : (int)$data["partneruserid"],
                "memberuserid" => empty($data["memberuserid"]) ? null : $data["memberuserid"],
            ];
        } catch (ClientException $e) {
            $res = $e->getResponse();

            Mage::log(sprintf(
                "AwarditAPI: POST %s%s (%s), response: %d %s",
                $host,
                $url,
                json_encode(array_intersect_key($body, ["loginkey" => true, "memberid" => true])),
                $res ? $res->getStatusCode() : 0,
                $res ? (string)$res->getBody() : "<empty>",
            ), Zend_Log::DEBUG, "awardit_points_api");

            if (($res ? $res->getStatusCode() : 0) === 400) {
                // Weirdo with bad request instead of unauthorized
                return null;
            }

            /** @var false|JavaError */
            $data = $this->decodeJson((string)$res->getBody());

            if (!$data) {
                throw $e;
            }

            return $data;
        }
    }

    /**
     * @param JavaSession $javaSession
     */
    public function getCustomerData(Mage_Core_Model_Store $store, array $javaSession): ?array
    {
        $helper = Mage::helper("awardit_points");
        $partnerId = $javaSession["partneruserid"] ?: $helper->getPartnerId($store);

        try {
            $res = $this->request($store, "GET", sprintf("partners/%s/member", $partnerId), [
                // Guzzle is weird with how cookies are handled
                "headers" => [
                    "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
                ],
            ]);

            return $this->decodeJson((string)$res->getBody());
        } catch (BadResponseException $e) {
            $res = $e->getResponse();

            if (($res ? $res->getStatusCode() : 0) === 404) {
                // 404 is expected
                return null;
            }

            throw $e;
        }
    }

    /**
     * @param JavaSession $javaSession
     * @return ApiPointsData
     */
    public function getPoints(Mage_Core_Model_Store $store, array $javaSession): array
    {
        $res = $this->request($store, "GET", "members/points", [
            // Guzzle is weird with how cookies are handled
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
        ]);

        return $this->decodeJson((string)$res->getBody());
    }


    /**
     * @param JavaSession $javaSession
     * @return array{ salesgroupId: ?string, salesgroupEmail: ?string }
     */
    public function getExtraInfo(Mage_Core_Model_Store $store, array $javaSession): array
    {
        $res = $this->request($store, "GET", "members/extrainfo", [
            // Guzzle is weird with how cookies are handled
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
        ]);

        $data = $this->decodeJson((string)$res->getBody());

        return [
            "salesgroupId" => empty($data["district"]["name"]) ? null : (string)$data["district"]["name"],
            "salesgroupEmail" => empty($data["district"]["email"]) ? null : (string)$data["district"]["email"],
        ];
    }

    public function addProductsFromOrder(
        Mage_Core_Model_Store $store,
        Mage_Sales_Model_Order $order
    ): void {
        $helper = Mage::helper("awardit_points");
        $factory = Mage::getModel("API/factory");
        $nullSer = Mage::getModel("API/serializer_constant", [ "value" => null ]);
        /** @var Crossroads_API_Model_Serializer_Decorator */
        $prodSer = Mage::getModel("awardit/product_decorator", [
            "exclude" => Crossroads_Awardit_Helper_Data::PRODUCT_PROPS_EXCLUDE,
            "inner" => $factory->createProductDetailSerializer($store)
                ->setCustomOptionsSerializer($nullSer)
                ->setRelatedSerializer($nullSer),
        ]);
        $collection = Mage::getModel("API/collection_product")
            ->setStore($store)
            ->setPage(1)
            ->setLimit(100)
            ->createCollection();

        $productIds = array_values(array_map(function (Mage_Sales_Model_Order_Item $item): string {
            return (string)$item->getProduct()->getId();
        }, array_filter($order->getAllItems(), function (Mage_Sales_Model_Order_Item $item): bool {
            return in_array($item->getProductType(), ["simple", "virtual"]);
        })));

        $collection->addAttributeToFilter("entity_id", ["in" => $productIds]);
        $collection->addAttributeToFilter("sku", ["nlike" => "awd_%"]);
        $collection->addAttributeToSort("entity_id", "ASC");

        $data = $prodSer->mapCollection($collection);
        $json = [
            "partneruserid" => $helper->getPartnerId($store),
            "products" => $data,
        ];

        $this->request($store, "POST", "/newproduct", [
            "json" => $json,
        ]);
    }

    /**
     * @param JavaSession $javaSession
     * @param Mage_Sales_Model_Order $order
     * @return ?CheckoutData
     */
    public function debitPoints(
        Mage_Core_Model_Store $store,
        array $javaSession,
        Mage_Sales_Model_Order $order
    ): ?array {
        $helper = Mage::helper("awardit_points");
        $listId = $helper->getListId($store);
        $session = Mage::getSingleton("customer/session");
        $customer = $order->getCustomer() ?? $session->getCustomer();

        assert($customer !== null);

        // Sync again to ensure the products exist
        //
        // We no longer synchronize the order products, they are only activated
        // after having been imported into the java monolith
        //
        // $this->addProductsFromOrder($store, $order);

        $products = array_values(
            array_filter(
                array_map(function (Mage_Sales_Model_Order_Item $item) use ($listId, $store, $helper) {
                    return $this->formatPointsProduct($store, $item, $listId);
                }, $order->getAllItems())
            )
        );

        $billing = $order->getBillingAddress();
        $shipping = $order->getShippingAddress();
        $points = $order->getPointsPoints();

        $pointsData = Awardit_Points_Model_PointsData::fromCustomer($customer);

        if ($pointsData->activeExact > 0) {
            $maxPoints = $pointsData->activeExact;

            if ($maxPoints < $points) {
                if ($maxPoints < $points - 1) {
                    throw new Exception(sprintf(
                        "%s: Order '%s' from quote %d is attempting to debit %d but only has %f",
                        __METHOD__,
                        $order->getIncrementId(),
                        $order->getQuoteId(),
                        $points,
                        $maxPoints
                    ));
                }

                Mage::log(sprintf(
                    "Adjusting order '%s' from quote %d to correct rouding error from %f to %d",
                    $order->getIncrementId(),
                    $order->getQuoteId(),
                    $maxPoints,
                    $points
                ), Zend_Log::DEBUG, "awardit-pointadjustment");

                $points = $maxPoints;
            }
        }

        $grandTotal = (float)$order->getGrandTotal();
        $tax = (float)$order->getTaxAmount();

        $json = [
            "orderid" => $order->getIncrementId(),
            "points" => $points,
            // Total amount of currency paid
            // TODO: Adjust when we get something besides SEK
            "sek" => $grandTotal,
            "sekExVat" => (float) $store->roundPrice($grandTotal - $tax),
            "products" => $products,
        ];

        if (! $billing) {
            throw new Exception(sprintf(
                "%s: Order '%s' from quote %d is missing a billing address",
                __METHOD__,
                $order->getIncrementId(),
                $order->getQuoteId()
            ));
        }

        if ($shipping) {
            $json["deliveryFirstname"] = $shipping->getFirstname();
            $json["deliveryLastname"] = $shipping->getLastname();
            $json["deliveryCity"] = $shipping->getCity();
            $json["deliveryZipCode"] = $shipping->getPostcode();
            $json["deliveryAdress"] = $shipping->getStreet();
        } else {
            $json["deliveryFirstname"] = $billing->getFirstname();
            $json["deliveryLastname"] = $billing->getLastname();
        }

        $json["deliveryEmail"] = $order->getCustomerEmail();

        // Check for awardit_point_source_restriction
        if ($order->getAwarditPointSourceRestriction()) {
            $json["onlypuid"] = $order->getAwarditPointSourceRestriction();
        }

        $res = $this->request($store, "POST", "/checkout", [
            // Guzzle is weird with how cookies are handled
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
            "json" => $json,
        ]);

        return $this->decodeJson((string)$res->getBody());
    }

    public function formatPointsProduct(
        Mage_Core_Model_Store $store,
        Mage_Sales_Model_Order_Item $item,
        string $listId
    ): ?array {
        if (
            !in_array($item->getProductType(), [
            Mage_Catalog_Model_Product_Type::TYPE_SIMPLE,
            Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL,
            ])
        ) {
            return null;
        }

        $helper = Mage::helper("awardit_points");
        $product = $item->getProduct();
        $entityId = $product->getId() ?: '';
        $sku = $product->getSku();

        if (strpos($sku, "awd_") === 0) {
            // Merchandiseid is numeric only, and custom options append value with a dash
            $merchId  = explode("-", substr($sku, 4), 2)[0];
            $customOptions = $item->getProductOptionByCode("options");

            $data = [
                "merchandiseid" => $merchId,
                "noofitems" => $item->getQtyOrdered(),
            ];

            // For a custom-option product, there should only be one value
            foreach ($customOptions ?: [] as $option) {
                $data["extra"] = $option["value"];
                break;
            }

            return $data;
        } elseif ($helper->isPriceMaster($store)) {
            // Magento provides prices instead of using internals in Java platform
            $taxClass = Mage::getModel('tax/class');
            $taxClass->load($product->getTaxClassId());
            return [
                "sku" => $sku,
                "priceinpoints" => round($item->getPointsRowValue() / $item->getQtyOrdered(), 0),
                "priceincash" => $product->getInvoicePrice(),
                "rawprice" => $product->getPurchasePrice(),
                "msrp" => $product->getMsrp(),
                "noofitems" => $item->getQtyOrdered(),
                "taxClass" => $taxClass->getClassName(),
                "taxPercent" => $item->getTaxPercent(),
            ];
        } else {
            return [
                "productid" => sprintf("cr_%s_%d", $entityId, $listId),
                "noofitems" => $item->getQtyOrdered(),
            ];
        }
    }

    public function resetPassword(Mage_Core_Model_Store $store, string $email): bool
    {
        $helper = Mage::helper("awardit_points");
        $locale = (string)$store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $json = [
            "langcode" => explode("_", $locale)[0],
            "email" => $email,
        ];
        try {
            $this->request($store, "POST", sprintf("/partners/%d/resetpassword", $helper->getPartnerId($store)), [
                "json" => $json,
            ]);

            return true;
        } catch (BadResponseException $e) {
            $res = $e->getResponse();

            if (($res ? $res->getStatusCode() : 0) === 400) {
                // 400 is expected
                return false;
            }

            throw $e;
        }
    }

    /**
     * @param JavaSession $javaSession
     */
    public function changePassword(
        Mage_Core_Model_Store $store,
        array $javaSession,
        string $oldPassword,
        string $newPassword
    ): bool {
        $helper = Mage::helper("awardit_points");
        $partnerId = $javaSession["partneruserid"] ?: $helper->getPartnerId($store);
        $json = [
            "oldpassword" => $oldPassword,
            "newpassword" => $newPassword,
            "confirmpassword" => $newPassword,
        ];

        try {
            $res = $this->request($store, "POST", sprintf("/partners/%s/changepassword", $partnerId), [
                "json" => $json,
                // Guzzle is weird with how cookies are handled
                "headers" => [
                    "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
                ],
            ]);

            $data = $this->decodeJson((string)$res->getBody());

            if (! is_array($data) || ! array_key_exists("code", $data) || $data["code"] !== 0) {
                Mage::log(sprintf("%s: Bad 200 response from changepassword: %s", __METHOD__, (string)$res->getBody()));

                return false;
            }

            return true;
        } catch (BadResponseException $e) {
            $res = $e->getResponse();

            if (($res ? $res->getStatusCode() : 0) === 400) {
                return false;
            }

            throw $e;
        }
    }

    /**
     * @param ?JavaSession $javaSession
     * @return ?array{texten:string, shownow:int}
     */
    public function getAgreement(Mage_Core_Model_Store $store, ?array $javaSession): ?array
    {
        $helper = Mage::helper("awardit_points");
        $partnerId = ($javaSession ? $javaSession["partneruserid"] : null) ?: $helper->getPartnerId($store);

        try {
            $res = $this->request($store, "GET", sprintf("/partners/%s/agreement", $partnerId), [
                // Guzzle is weird with how cookies are handled
                "headers" => $javaSession ? [
                    "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
                ] : [],
            ]);

            return $this->decodeJson((string)$res->getBody());
        } catch (BadResponseException $e) {
            $res = $e->getResponse();

            if (($res ? $res->getStatusCode() : 0) === 404) {
                return null;
            }

            throw $e;
        }
    }

    /**
     * @param JavaSession $javaSession
     * @return Array<string, AccountConfig>
     */
    private function getAccountConfig(
        Mage_Core_Model_Store $store,
        array $javaSession
    ): array {
        $helper = Mage::helper("awardit_points");
        $partnerId = $javaSession["partneruserid"] ?: $helper->getPartnerId($store);
        // Some defaults
        $fields = [
            "fname"    => "required",
            "lname"    => "required",
            "email"    => "required",
            "cell"     => "required",
            "comp"     => "required",
            "orgnr"    => "required",
            "tmih"     => "hidden",
            "street"   => "required",
            "zip"      => "required",
            "city"     => "required",
            "care"     => "required",
            //"care2"    => "required", // added later
            "pnr"      => "hidden",
            "birth"    => "hidden",
            "gender"   => "hidden",
            "muid"     => "hidden",
            "passw"    => "required",
            "retailer" => "optional",
            "reg3"     => "hidden",
            "country"  => "hidden",

            // Added in v1.4.3
            "regioncode2"      => "optional",
            "district"         => "optional",
            "totmemberinhouse" => "optional",
            "careof"           => "optional",
            "care2"            => "optional",
            "ssn"              => "optional",
        ];

        try {
            $res = $this->request($store, "GET", sprintf("/partners/%s/accountconfig", $partnerId), [
                "headers" => [
                    "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
                ],
            ]);

            /**
             * This is either empty, or contains a list of objects named after the field it represents.
             * This is an example:
             *
             * * "birth":{
             * *     "name":"Fødselsdato (ÅÅÅÅ-MM-DD)",
             * *     "val":2,
             * *     "ro":"",
             * *     "req":"",
             * *     "reqs":"",
             * *     "show":true,
             * *     "min":10,
             * *     "max":10,
             * *     "digits":false
             * * }
             *
             * The `val` attribute have the following possible values:
             *
             * * 0: Required
             * * 1: Read-only
             * * 2: Optional
             * * 3: hidden
             *
             * @var ?Array<string, int|string|bool>
             */
            $data = $this->decodeJson((string)$res->getBody());

            foreach ($data ?? [] as $fieldName => $properties) {
                if (!array_key_exists("val", $properties)) {
                    continue;
                }

                // Default
                $val = "optional";

                switch ($properties["val"]) {
                    case 0:
                        $val = "required";
                        break;
                    case 1:
                        $val = "readonly";
                        break;
                    case 2:
                        $val = "optional";
                        break;
                    case 3:
                        $val = "hidden";
                        break;
                }

                $fields[$fieldName] = $val;
            }
        } catch (Exception $e) {
            // Warn since we do not want to prevent saving customers, and this
            // data can vary a lot.
            Mage::logException($e, "awardit_points_api", Zend_Log::WARN);
        }

        return $fields;
    }

    /**
     * @param Mage_Customer_Model_Customer $customer
     * @param JavaSession $javaSession
     */
    public function updateCustomer(
        Mage_Customer_Model_Customer $customer,
        Mage_Core_Model_Store $store,
        array $javaSession,
        Mage_Customer_Model_Address_Abstract $address
    ): void {
        $fields = $this->getAccountConfig($store, $javaSession);
        $helper = Mage::helper("awardit_points");
        $partnerId = $javaSession["partneruserid"] ?: $helper->getPartnerId($store);

        $street = $address->getStreet();
        $dob = $customer->getDob();
        $genderId = $customer->getGender();
        $json = [
            "firstname" => $address->getFirstname(),
            "lastname" => $address->getLastname(),
            "streetaddress" => is_array($street) ? implode(",", $street) : $street,
            "company" => $address->getCompany() ?: null,
            "streetcode" => $address->getCity(),
            "zipcode" => $address->getPostcode(),
            "cellphonenum" => $address->getTelephone(),
            "phone" => $address->getTelephone(),
            "organisationsnummer" => $address->getAwarditOrgNr(),
        ];

        // Added in v1.4.3
        $attributeList = [
            // magento attribute       => Java attribute
            "awardit_retailer"         => "regioncode2",
            "awardit_district"         => "district",
            "awardit_totmemberinhouse" => "totmemberinhouse",
            "awardit_customer_extra"   => "careof",
            "awardit_customer_extra2"  => "care2",
            "awardit_ssn"              => "ssn",
        ];
        foreach ($attributeList as $attributeCode => $translatedCode) {
            if (in_array($fields[$translatedCode], ["optional", "required"])) {
                $json[$translatedCode] = $customer->getData($attributeCode);
            }
        }

        // Only write the birthdate if it is optional or required
        if (! empty($dob) && in_array(($fields["birth"] ?? "hidden"), ["optional", "required"])) {
            $date = new DateTime($dob);

            $json["birthyear"] = $date->format("Y");
            $json["birthmonth"] = $date->format("m");
            $json["birthday"] = $date->format("d");
        }

        if (! is_null($genderId)) {
            $genderAttr = Mage::getResourceModel("customer/customer")
                ->getAttribute("gender");

            if ($genderAttr) {
                $genderText = $genderAttr->getSource()
                    ->getOptionText($genderId);

                $genderKey = array_search($genderText, Awardit_Points_Model_Customer::GENDER_MAP, true);

                if ($genderKey) {
                    $json["gender"] = $genderKey;
                }
            }
        }

        if (in_array($customer->getAwarditMailnotify(), ["175", "176"])) {
            $json["mailnotify"] = $customer->getAwarditMailnotify();
        }

        if (! preg_match("/@example\.com$/i", $customer->getEmail())) {
            $json["email"] = $customer->getEmail();
        }

        // PUT works like PATCH here
        $this->request($store, "PUT", sprintf("/partners/%s/member", $partnerId), [
            "json" => $json,
            // Guzzle is weird with how cookies are handled
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
        ]);
    }

    /**
     * @param JavaSession $javaSession
     */
    public function activatePartner(
        Mage_Core_Model_Store $store,
        array $javaSession,
        string $id
    ): bool {
        $options = [
            "json" => [
                "afid" => $id,
            ],
            // Guzzle is weird with how cookies are handled
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
        ];

        $res = $this->request($store, "POST", sprintf("/members/activatepartner"), $options);
        $data = $this->decodeJson((string)$res->getBody());

        if (is_array($data) && ! empty($data["result"])) {
            return true;
        }

        return false;
    }

    /**
     * @param JavaSession $javaSession
     */
    public function agreeToAgreement(Mage_Core_Model_Store $store, array $javaSession): bool
    {
        $helper = Mage::helper("awardit_points");
        $partnerId = $javaSession["partneruserid"] ?: $helper->getPartnerId($store);

        // Super weird, why is this a GET?!?
        $res = $this->request($store, "GET", sprintf("/partners/%s/agree", $partnerId), [
            "headers" => [
                "Cookie" => "JSESSIONID=" . $javaSession["JSESSIONID"],
            ],
        ]);

        return ($this->decodeJson((string)$res->getBody())["agree"] ?? 0) > 0;
    }

    public function requestAccountVerificationEmail(Mage_Core_Model_Store $store, string $memberuserid): bool
    {
        $helper = Mage::helper("awardit_points");
        $partnerId = $helper->getPartnerId($store);
        $body = [
            "memberid" => $memberuserid,
        ];

        $res = $this->request($store, "POST", sprintf("/partners/%s/requestverify", $partnerId), [
            "json" => $body,
        ]);

        return $this->decodeJson((string)$res->getBody())["emailsent"] ?? false;
    }
}
