<?php

declare(strict_types=1);

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_Points_Model_Observer extends Mage_Core_Model_Abstract
{
    /**
     * Verifies unique attributes for customer on save.
     */
    public function customerSaveBefore(Varien_Event_Observer $observer): void
    {
        /**
         * @var Mage_Customer_Model_Customer
         */
        $customer = $observer->getCustomer();

        foreach ($customer->getAttributes() as $attribute) {
            if (! $attribute->getIsUnique()) {
                continue;
            }
            $value = $customer->getData($attribute->getAttributeCode());

            if (empty($value) || strlen(trim($value)) === 0) {
                continue;
            }

            $collection = Mage::getModel("customer/customer")
                ->getCollection()
                ->addAttributeToFilter("website_id", $customer->getWebsiteId())
                ->addAttributeToFilter($attribute->getAttributeCode(), $value)
                ->addFieldToFilter("entity_id", ["neq" => $customer->getId()]);

            if ($collection->getSize() > 0) {
                Mage::throwException(sprintf(
                    "The value '%s' for %s is already used in this website",
                    $value,
                    $attribute->getStoreLabel()
                ));
            }
        }
    }

    /**
     * Observer listening to `sales_order_invoice_save_after` to update deposit amount invoiced in order.
     */
    public function invoiceSaveAfter(Varien_Event_Observer $observer): void
    {
        $invoice = $observer->getInvoice();
        $order = $invoice->getOrder();

        if ($invoice->getBaseAwarditAmount()) {
            $order->setAwarditAmountInvoiced(
                $order->getAwarditAmountInvoiced() + $invoice->getAwarditAmount()
            );
            $order->setBaseAwarditAmountInvoiced(
                $order->getBaseAwarditAmountInvoiced() + $invoice->getBaseAwarditAmount()
            );
        }
    }

    /**
     * Observer listening to `sales_order_creditmemo_save_after` to update deposit amount refunded in order.
     */
    public function creditmemoSaveAfter(Varien_Event_Observer $observer): void
    {
        $creditmemo = $observer->getCreditmemo();
        $order = $creditmemo->getOrder();

        if ($creditmemo->getBaseAwarditAmount()) {
            $order->setAwarditAmountRefunded(
                $order->getAwarditAmountRefunded() + $creditmemo->getAwarditAmount()
            );
            $order->setBaseAwarditAmountRefunded(
                $order->getBaseAwarditAmountRefunded() + $creditmemo->getBaseAwarditAmount()
            );
        }
    }

    public function blacklistEmails(Mage_Sales_Model_Quote $quote): void
    {
        $email = $quote->getCustomerEmail();

        if (empty($email) || preg_match("/@example\.com$/i", $email)) {
            throw new Awardit_Points_Exception_BlacklistedEmailException();
        }
    }

    public function updateCustomer(Varien_Event_Observer $observer): void
    {
        $customer = $observer->getCustomer();
        /**
         * @var MageQL_Core_Model_Context
         */
        $context = $observer->getContext();

        if (! $customer->getAwarditMemberId()) {
            return;
        }

        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
            Mage::log(sprintf(
                "Missing JavaSession for customer %d (member %s)",
                $customer->getId(),
                $customer->getAwarditMemberId() ?: "<none>"
            ), Zend_Log::ERR, "awardit_points");

            return;
        }

        $billing = $customer->getDefaultBillingAddress();

        if (! $billing) {
            Mage::log(sprintf(
                "Missing Billing address for customer %d (member %s)",
                $customer->getId(),
                $customer->getAwarditMemberId() ?: "<none>"
            ), Zend_Log::WARN, "awardit_points");

            return;
        }

        $api->updateCustomer($customer, $context->getStore(), $javaSession, $billing);
    }

    /**
     * Observer listening to `sales_order_place_after` to run provider code.
     */
    public function salesOrderPlaceAfter(Varien_Event_Observer $observer): void
    {
        $order = $observer->getOrder();

        // Handle >0 in standard Points core observer
        if (abs($order->getBasePointsAmount()) >= 0.1) {
            return;
        }

        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore($order->getStoreId());
        $quote = $order->getQuote();
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
             Mage::log(sprintf(
                 "Missing JavaSession for customer %d and quote %s",
                 $order->getAwarditMemberId(),
                 $quote->getId()
             ), Zend_Log::ERR, "awardit_points");
            return;
        }

        try {
            $response = $api->debitPoints($store, $javaSession, $order);
            $pointSources = $response['pointSources'] ?: null;
            $order->setAwarditPointSources(json_encode($pointSources));
        } catch (Exception $e) {
            Mage::logException($e);
        }
    }

    public function appendAdminHtml(Varien_Event_Observer $event): void
    {
        $block = $event->getBlock();
        if ($block->getNameInLayout() != 'order_payment') {
            return;
        }
        $child = $block->getChild('awarditpoints.source');
        $transport = $event->getTransport();
        if ($child && $transport) {
            $transport->setHtml("{$transport->getHtml()}{$child->toHtml()}");
        }
    }
}
