<?php

declare(strict_types=1);

/**
 * @psalm-type SSOData array{
 *   cellphonenum?: string,
 *   firstname?: string,
 *   lastname?: string,
 *   zipcode?: string,
 *   streetcode?: string,
 *   company?: string,
 *   streetaddress?: string,
 *   memberuserid: string,
 *   email: string,
 *   organisationsnummer?: string,
 *   activepoints?: string,
 *   totmemberinhouse?: string,
 *   mailnotify?: string,
 *   usertype?: string,
 *   countrycode?: string,
 *   anonymized_id?: string,
 *   careof?: string,
 *   regioncode1?: string,
 *   regioncode2?: string,
 *   regioncode3?: string,
 *   personnummer?: string,
 * }
 * @psalm-type ExtraInfoData array{ salesgroupId: ?string, salesgroupEmail: ?string }
 * @psalm-import-type ApiPointsData from Awardit_Points_Model_Api
 * @psalm-import-type JavaSession from Awardit_Points_Model_Api
 * @psalm-import-type EmailVerify from Awardit_Points_Model_Api
 * @psalm-import-type JavaError from Awardit_Points_Model_Api
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_Points_Model_Customer extends Mage_Core_Model_Abstract
{
    public const CONFIG_ALLOWED_COUNTRIES = "general/country/allow";
    public const SUCCESS = "success";
    public const ERROR = "error";
    public const NOT_MODIFIED = "notModified";
    public const ERROR_NOT_LOGGED_IN = "errorNotLoggedIn";
    public const ERROR_LOGIN_FAILED = "errorLoginFailed";
    public const ERROR_MISSING_CUSTOMER = "errorMissingCustomer";
    public const EMAILVERIFY_TYPE_REQUEST = "requestVerification";
    public const EMAILVERIFY_TYPE_SENT = "verificationEmailSent";
    public const EMAILVERIFY_TWOFACTOR = "twofactor";
    /**
     * Length of random characters used to create a placeholder email if no
     * awardit member id is present.
     */
    public const RANDOM_LENGTH = 8;
    /**
     * Map of the gender-ids relationship between magento and awardit-backend
     * Id 165, 166 and 167 are used in awardit-backend, while Male, Female and Unspecified
     * are used in magento.
     */
    public const GENDER_MAP = [165 => "Male", 166 => "Female", 167 => "Unspecified"];

    public const MAILNOTIFY_YES = 175;
    public const MAILNOTIFY_NO = 176;

    public static function resolveAwardit(
        Mage_Customer_Model_Customer $customer,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): Varien_Object {
        $helper = Mage::helper("awardit_points");
        $session = Mage::getSingleton("customer/session");
        $javaSession = $helper->getJavaSession();
        $partneruserid = $session->getJavaPartneruserid();

        if (empty($partneruserid)) {
            $partneruserid = $helper->getPartnerId($ctx->getStore());
        }

        if ($javaSession && $session->getCustomer() === $customer) {
            Mage::getModel("awardit_points/customer")->updatePoints($ctx->getStore(), $customer, $javaSession);
        }

        $points = Awardit_Points_Model_PointsData::fromCustomer($customer);

        return new Varien_Object([
            "member_id" => $customer->getAwarditMemberId() ?: null,
            // $points->active rounds up, which causes user to make purchase with more points than they have
            "active_points" => floor($points->activeExact),
            "expiring_points" => $points->expiring,
            "expiring_points_date" => $points->expiredate,
            "rank" => $points->rank,
            "pending_points" => $points->pending,
            "group_points" => $points->group,
            "group_rank" => $points->groupRank,
            "own_points" => $points->own,
            "own_rank" => $points->ownRank,
            "waiting_points" => $points->waitingPoints,
            "min_threshold" => $points->minThreshold,
            "earned_points" => $points->earnedPoints,
            "rate" => $helper->getRate($ctx->getStore()),
            "number_of_employees" => (int)$customer->getAwarditTotmemberinhouse(),
            "mailnotify" => ((int)$customer->getAwarditMailnotify()) === self::MAILNOTIFY_YES,
            "user_type" => in_array($customer->getAwarditUsertype(), [null, ""], true)
                ? null
                : (int)$customer->getAwarditUsertype(),
            "partneruserid" => (int)$partneruserid,
            "agreement_accepted" => (bool)$customer->getAwarditAgreementAccepted(),
        ]);
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function resolveAgreement(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): ?Varien_Object {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        $javaSession = $helper->getJavaSession();

        $data = $api->getAgreement($store, $javaSession);

        if (! $data) {
            return null;
        }

        $content = Htmlawed::filter($data["texten"], [
            "safe" => 1, // Make the input safe, considers input as untrusted
            "deny_attribute" => "style",
        ]);

        return new Varien_Object([
            "content" => $content,
        ]);
    }

    /**
     * @param ?JavaSession|EmailVerify|JavaError $javaSession
     * @return array{
     *   result:string,
     *   customer:?Mage_Customer_Model_Customer,
     *   memberuserid:?string,
     *   mailto:?string,
     *   twofactorSkipDays:?int,
     *   twofactorWaitAmount:?int,
     *   twofactorWaitType:?int,
     *   twofactorWaitTypeString:?string,
     * }
     */
    private static function handleSession(Mage_Core_Model_Store $store, ?array $javaSession): array
    {
        if (! $javaSession) {
            return [
                "result" => self::ERROR_LOGIN_FAILED,
                "customer" => null,
                "memberuserid" => null,
                "mailto" => null,
            ];
        }

        if (! empty($javaSession["code"])) {
            /** @var JavaError $javaSession */
            return [
                "result" => self::ERROR_LOGIN_FAILED,
                "customer" => null,
                "memberuserid" => null,
                "mailto" => null,
                "twofactorSkipDays" => null,
                "twofactorWaitAmount" => $javaSession["tf_wait"],
                "twofactorWaitType" => $javaSession["tf_type"],
                "twofactorWaitTypeString" => $javaSession["tf_type_str"],
            ];
        }

        if (! empty($javaSession["emailverify"])) {
            /** @var EmailVerify $javaSession */
            return [
                "result" => $javaSession["emailverify"],
                "JSESSIONID" => $javaSession["JSESSIONID"],
                "customer" => null,
                "memberuserid" => $javaSession["memberuserid"],
                "mailto" => $javaSession["mailto"] ?? null,
                "twofactorSkipDays" => $javaSession["twofactorSkipDays"] ?? null,
                "twofactorWaitAmount" => null,
                "twofactorWaitType" => null,
                "twofactorWaitTypeString" => null,
            ];
        }

        /** @var JavaSession $javaSession */
        $customer = self::_getCustomerData($store, $javaSession);

        if (! $customer) {
            return [
                "result" => self::ERROR_MISSING_CUSTOMER,
                "customer" => null,
                "memberuserid" => null,
                "mailto" => null,
                "twofactorSkipDays" => null,
                "twofactorWaitAmount" => null,
                "twofactorWaitType" => null,
                "twofactorWaitTypeString" => null,
            ];
        }

        return [
            "result" => self::SUCCESS,
            "customer" => self::_getCustomerData($store, $javaSession),
            "memberuserid" => $javaSession["memberuserid"],
            "mailto" => null,
            "twofactorSkipDays" => null,
            "twofactorWaitAmount" => null,
            "twofactorWaitType" => null,
            "twofactorWaitTypeString" => null,
        ];
    }

    /**
     * @param mixed $unusedSrc
     * @param array{username:string, password:string} $args
     * @return array{result:string, customer:?Mage_Customer_Model_Customer, memberuserid:?string}
     */
    public static function mutateLogin($unusedSrc, array $args, MageQL_Core_Model_Context $ctx): array
    {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");
        $javaSession = null;

        try {
            $javaSession = $api->loginCustomer($store, $args["username"], $args["password"]);
        } catch (Exception $e) {
            Mage::log(
                sprintf(
                    "Request to AwarditAPI failed. Could not login user: %s. Error: %s",
                    $args["username"],
                    $e->getMessage()
                ),
                Zend_Log::ERR,
                "awardit_points_api"
            );
        }

        return self::handleSession($store, $javaSession);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{code:string, remember:bool} $args
     * @return array{result:string, customer:?Mage_Customer_Model_Customer, memberuserid:?string}
     */
    public static function mutateLoginOtp($unusedSrc, array $args, MageQL_Core_Model_Context $ctx): array
    {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");
        $javaSession = null;

        try {
            $javaSession = $api->loginCustomerOtp($store, $args["code"], $args["remember"]);
        } catch (Exception $e) {
            Mage::log(
                sprintf(
                    "Request to AwarditAPI failed. Could not login user with otp. Error: %s",
                    $e->getMessage()
                ),
                Zend_Log::ERR,
                "awardit_points_api"
            );
        }

        return self::handleSession($store, $javaSession);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{loginkey:string, hiddenlogin:string} $args
     * @return array{result:string, customer:?Mage_Customer_Model_Customer, memberuserid:?string, mailto:?string}
     */
    public static function mutateLoginAsCustomer($unusedSrc, array $args, MageQL_Core_Model_Context $ctx)
    {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");
        $javaSession = null;

        try {
            $javaSession = $api->loginAsCustomer($store, $args["loginkey"], $args["hiddenlogin"]);
        } catch (Exception $e) {
            Mage::log(
                sprintf("Request to AwarditAPI failed. Error: %s", $e->getMessage()),
                Zend_Log::ERR,
                "awardit_points_api"
            );
        }

        return self::handleSession($store, $javaSession);
    }

    /**
     * @param Mage_Core_Model_Store $store,
     * @param JavaSession $javaSession
     * @return ?Mage_Customer_Model_Customer
     */
    private static function _getCustomerData(Mage_Core_Model_Store $store, array $javaSession)
    {
        // Reset the cart to avoid issues caused by merging carts in the case
        // of login as customer or multiple partners mapping to a single store
        $checkout = Mage::getSingleton("checkout/session");

        $checkout->clear();
        $checkout->clearHelperData();

        $session = Mage::getSingleton("customer/session");
        $api = Mage::getSingleton("awardit_points/api");

        $customerData = $api->getCustomerData($store, $javaSession);

        if (! $customerData) {
            return null;
        }

        $pointsData = $api->getPoints($store, $javaSession);
        $extraInfoData = $api->getExtraInfo($store, $javaSession);

        $customer = self::fetchCustomer($customerData, $pointsData, $extraInfoData, $store);

        Mage::log(sprintf(
            "%s: Logged in awardit customer %s as %s (entityId: %d)",
            __METHOD__,
            $customerData["id"],
            $customer->getEmail(),
            $customer->getId() ?: ''
        ));

        $session->setCustomerAsLoggedIn($customer);
        $session->setJavaSession($javaSession["JSESSIONID"]);
        $session->setJavaDeviceSession($javaSession["Cookie_2fa"]);
        $session->setJavaPartneruserid($javaSession["partneruserid"]);
        $session->setJavaMemberuserid($javaSession["memberuserid"]);

        return $customer;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{email: string} $args
     */
    public static function mutateResetPassword(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): string {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");

        if ($api->resetPassword($store, $args["email"])) {
            return self::SUCCESS;
        }

        return self::ERROR;
    }

    /**
     * @param mixed $unusedSrc
     * @param array{oldPassword:string, newPassword:string} $args
     * @return self::SUCCESS|self::ERROR|self::ERROR_NOT_LOGGED_IN
     */
    public static function mutateChangePassword(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $store = $ctx->getStore();
        $helper = Mage::helper("awardit_points");
        $api = Mage::getSingleton("awardit_points/api");
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
            return self::ERROR_NOT_LOGGED_IN;
        }

        if ($api->changePassword($store, $javaSession, $args["oldPassword"], $args["newPassword"])) {
            return self::SUCCESS;
        }

        return self::ERROR;
    }

    /**
     * @param mixed $unusedSrc
     * @return self::SUCCESS|self::ERROR|self::ERROR_NOT_LOGGED_IN
     */
    public static function mutateSetMailnotify(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ) {
        $store = $ctx->getStore();
        $helper = Mage::helper("awardit_points");
        $api = Mage::getSingleton("awardit_points/api");
        $session = Mage::getSingleton("customer/session");
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
            return self::ERROR_NOT_LOGGED_IN;
        }

        $customer = $session->getCustomer();

        $customer->setAwarditMailnotify(
            $args["value"] === true ?
            (string)self::MAILNOTIFY_YES :
            (string)self::MAILNOTIFY_NO
        );
        /** @var Mage_Customer_Model_Address|false $billing */
        $billing = $customer->getDefaultBillingAddress();

        if ($billing === false) {
            Mage::log(sprintf(
                "Missing Billing address for customer %d (member %s)",
                $customer->getId() ?: '',
                $customer->getAwarditMemberId() ?: "<none>"
            ));

            return self::ERROR;
        }

        $api->updateCustomer($customer, $store, $javaSession, $billing);

        $customer->save();

        return self::SUCCESS;
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateAgreementAgree(
        $unusedSrc,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): string {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        $javaSession = $helper->getJavaSession();
        $session = Mage::getSingleton("customer/session");
        $customer = $session->getCustomer();

        if (! $javaSession || ! $customer->getId()) {
            return self::ERROR_NOT_LOGGED_IN;
        }

        if ($api->agreeToAgreement($store, $javaSession)) {
            $customer->setAwarditAgreementAccepted(true);
            $customer->save();

            return self::SUCCESS;
        }

        return self::NOT_MODIFIED;
    }

    /**
     * Feches a customer from magento matching the SSO-data, if no such customer exists
     * a new customer will be created.
     *
     * @param SSOData $ssoData
     * @param ApiPointsData $pointsData
     * @param ExtraInfoData $extraInfoData
     * @return Mage_Customer_Model_Customer
     */
    public static function fetchCustomer(
        array $ssoData,
        array $pointsData,
        array $extraInfoData,
        Mage_Core_Model_Store $store
    ): Mage_Customer_Model_Customer {
        $email = trim($ssoData["email"]);
        $userId = $ssoData["memberuserid"];

        if (empty($email)) {
            $email = sprintf("%s@example.com", $userId);
        }

        // First attempt to load a customer with the supplied email
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail($email);

        if ($customer->getId() && $customer->getAwarditMemberId() === $userId) {
            // Email and user identifier matches
            self::updateData($customer, $ssoData, $pointsData, $extraInfoData, $store);

            return $customer;
        }

        if ($customer->getId()) {
            // We have a customer with duplicate email but the wrong identifier
            self::renameCustomer($customer);

            // Reset customer
            $customer = Mage::getModel("customer/customer");
        }

        // Try to find customer by AwarditMemberId
        $collection = $customer->getCollection()
            ->addAttributeToFilter("website_id", $store->getWebsiteId())
            ->addAttributeToFilter("awardit_member_id", $userId)
            ->load();

        if ($collection->count() > 1) {
            throw new Exception(sprintf(
                "%s: User identifier '%s' matches more than one customer in website '%s'.",
                __METHOD__,
                $userId,
                $store->getWebsiteId() ?: ""
            ));
        }

        // This will either give us a customer, or give us an empty customer object
        $customer = $collection->getFirstItem();

        if (! $customer->getEmail()) {
            Mage::log(sprintf(
                "%s: Creating awardit customer %s",
                __METHOD__,
                json_encode($ssoData, JSON_INVALID_UTF8_IGNORE | JSON_UNESCAPED_UNICODE)
            ));

            $customer->setEmail($email);
        } else {
            Mage::log(sprintf(
                "%s: Updating awardit customer %s (entityId: %d) with %s",
                __METHOD__,
                $customer->getEmail(),
                $customer->getId(),
                json_encode($ssoData, JSON_INVALID_UTF8_IGNORE | JSON_UNESCAPED_UNICODE)
            ));
        }

        // Should be free now
        $customer->setEmail($email);
        $customer->setAwarditMemberId($userId);

        self::updateData($customer, $ssoData, $pointsData, $extraInfoData, $store);

        return $customer;
    }

    /**
     * Renames a duplicate email.
     *
     * @param Mage_Customer_Model_Customer $customer
     */
    protected static function renameCustomer(Mage_Customer_Model_Customer $customer): void
    {
        list($mailbox, $domain) = explode("@", $customer->getEmail());

        $userId = $customer->getAwarditMemberId();

        if (! $userId) {
            $userId = Mage::helper("core")->getRandomString(self::RANDOM_LENGTH);
        }

        if (($pos = strpos($mailbox, "+")) !== false) {
            // We already have a tag, copy it to the userId
            $userId .= "_" . substr($mailbox, $pos + 1);
            $mailbox = substr($mailbox, 0, $pos);
        }

        $customer->setEmail(sprintf("%s+%s@%s", $mailbox, $userId, $domain));
        $customer->save();
    }

    /**
     * @param Mage_Customer_Model_Customer $customer
     * @param SSOData $ssoData
     * @param ApiPointsData $pointsData
     * @param ExtraInfoData $extraInfoData
     */
    protected static function updateData(
        Mage_Customer_Model_Customer $customer,
        array $ssoData,
        array $pointsData,
        array $extraInfoData,
        Mage_Core_Model_Store $store
    ): void {
        $helper = Mage::helper("awardit_points");

        $customer->setWebsiteId($store->getWebsiteId());

        $customer->setFirstname($ssoData["firstname"] ?? "");
        $customer->setLastname($ssoData["lastname"] ?? "");

        if (array_key_exists("gender", $ssoData)) {
            $gender = (int) $ssoData["gender"];

            if (array_key_exists($gender, self::GENDER_MAP)) {
                $genderAttr = Mage::getResourceModel("customer/customer")
                    ->getAttribute('gender');

                if ($genderAttr) {
                    $genderId = $genderAttr->getSource()
                        ->getOptionId(self::GENDER_MAP[$gender]);

                    $customer->setGender($genderId ? (int)$genderId : null);
                }
            }
        }

        $helper->setCustomerPoints($customer, $pointsData);

        // 1900-01-01 and earlier are placeholders, corresponding to "no date provided"
        $dob = !empty($ssoData["dateofbirth"]) && $ssoData["dateofbirth"] > '1900-01-01'
            ? $ssoData["dateofbirth"]
            : "";
        $customer->setDob($dob);

        $customer->setAwarditSalesgroupId($extraInfoData["salesgroupId"] ?? "");
        $customer->setAwarditSalesgroupEmail($extraInfoData["salesgroupEmail"] ?? "");

        $customer->setAwarditAnonymizedId($ssoData["anonymized_id"] ?? null);

        $customer->setAwarditAgreementAccepted((bool)($ssoData["securitystatus"] ?? 0));

        $customer->setAwarditVisitNumber(array_key_exists("visitnumber", $ssoData) ?
            (int)$ssoData["visitnumber"] :
            null);
        $customer->setAwarditTotmemberinhouse(
            array_key_exists("totmemberinhouse", $ssoData) ?
            $ssoData["totmemberinhouse"] :
            null
        );
        $customer->setAwarditMailnotify(
            array_key_exists("mailnotify", $ssoData) ?
            $ssoData["mailnotify"] :
            null
        );
        $customer->setAwarditUsertype(
            array_key_exists("usertype", $ssoData) ?
            $ssoData["usertype"] :
            null
        );

        $customer->setAwarditCustomerExtra($ssoData["careof"] ?? "");
        $customer->setAwarditCustomerExtra2($ssoData["regioncode1"] ?? "");
        $customer->setAwarditDistrict($ssoData["regioncode3"] ?? "");
        $customer->setAwarditSsn($ssoData["personnummer"] ?? "");
        $customer->setAwarditRetailer($ssoData["regioncode2"] ?? "");

        $customer->setAwarditTotchildinhouse((int)($ssoData["totchildinhouse"] ?? "0"));

        self::replaceAddresses($customer, $ssoData);

        $customer->save();
    }

    /**
     * @param Mage_Customer_Model_Customer $customer
     * @param SSOData $ssoData
     */
    protected static function replaceAddresses(
        Mage_Customer_Model_Customer $customer,
        array $ssoData
    ): void {
        // Delete old addresses
        foreach ($customer->getAddresses() as $addr) {
            $addr->delete();
        }

        // Make sure the customer object has no addresses left, otherwise they will be recreated
        // as empty addresses
        $customer->cleanAllAddresses();

        // Need to save the customer before we create the address to make sure we have a customer id
        $customer->save();

        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore();
        $address = Mage::getModel("customer/address");

        $address->setCustomerId($customer->getId());
        $address->setFirstname($customer->getFirstname());
        $address->setLastname($customer->getLastname());
        $address->setStreet(trim($ssoData["streetaddress"] ?? ""));
        $address->setCompany($ssoData["company"] ?? "");
        $address->setAwarditOrgNr($ssoData["organisationsnummer"] ?? "");
        $address->setCity($ssoData["streetcode"] ?? "");
        $address->setPostcode(self::stripNonASCII($ssoData["zipcode"] ?? ""));
        $address->setTelephone(self::stripNonASCII($ssoData["cellphonenum"] ?? ""));

        $countryId = strtoupper((string)($ssoData["countrycode"] ?? "SE"));

        if (! self::isValidCountryId($countryId, $store)) {
            Mage::log(sprintf(
                "Invalid country id supplied from SSO: '%s', falling back to default country.",
                $countryId
            ), Zend_Log::WARN, "awardit_points");

            $countryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

            if (! $countryId) {
                throw new Exception(sprintf(
                    "%s: The default country is not set for store '%s'.",
                    __METHOD__,
                    $store->getCode()
                ));
            }
        }

        $address->setCountryId($countryId);

        $address->setIsDefaultShipping(true);
        $address->setIsDefaultBilling(true);

        $address->save();
        $addressId = (int)$address->getId();

        $customer->addAddress($address);
        $customer->setDefaultBilling($addressId);
        $customer->setDefaultShipping($addressId);
    }

    public static function stripNonASCII(string $str): string
    {
        return trim(preg_replace("/[[:^ascii:]]/", "", $str));
    }

    /**
     * @param Mage_Customer_Model_Customer $customer
     * @param JavaSession $javaSession
     */
    public function updatePoints(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer,
        array $javaSession
    ): void {
        $helper = Mage::helper("awardit_points");
        $api = Mage::getSingleton("awardit_points/api");

        try {
            $pointsData = $api->getPoints($store, $javaSession);

            $helper->setCustomerPoints($customer, $pointsData);

            $customer->save();
        } catch (Exception $e) {
            Mage::logException($e);
        }
    }

    public static function isValidCountryId(
        ?string $countryId,
        Mage_Core_Model_Store $store
    ): bool {
        if (!$countryId) {
            return false;
        }

        $approvedIds = array_map(
            "strtoupper",
            array_filter(explode(",", $store->getConfig(self::CONFIG_ALLOWED_COUNTRIES) ?: ""))
        );

        return in_array($countryId, $approvedIds);
    }

    /**
     * @param mixed $unusedSrc
     * @param array{memberuserid:string} $args
     */
    public static function mutateRequestAccountVerificationEmail(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): bool {
        $store = $ctx->getStore();
        $api = Mage::getSingleton("awardit_points/api");

        return $api->requestAccountVerificationEmail($store, $args["memberuserid"]);
    }
}
