<?php

declare(strict_types=1);

use Points\Core\ProviderInterface;

class Awardit_Points_Model_Provider implements ProviderInterface
{
    /**
     * Returns true if this point provider is enabled for this store.
     */
    public function isEnabled(Mage_Core_Model_Store $store): bool
    {
        return Mage::helper("awardit_points")->isEnabled($store);
    }

    /**
     * Returns true if this point payment provider can be used with the supplied quote.
     */
    public function appliesTo(Mage_Sales_Model_Quote $quote): bool
    {
        return Mage::helper("awardit_points")->isEnabled($quote->getStore());
    }

    /**
     * Returns the label show to the customer for this point type.
     */
    public function getLabel(Mage_Core_Model_Store $store): string
    {
        return Mage::helper("awardit_points")->getLabel($store);
    }

    /**
     * Returns the amount of points the shipping would cost for the given address.
     */
    public function getQuoteShippingPrice(Mage_Sales_Model_Quote_Address $address): ?int
    {
        $helper = Mage::helper("awardit_points");
        $quote = $address->getQuote();
        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore($quote->getStoreId());

        $rate = $helper->getRate($store);

        if (! $rate) {
            return null;
        }

        $amount = $address->getBaseShippingAmount() + $address->getBaseShippingTaxAmount();

        $value = (int)ceil($amount * $rate);

        return $value;
    }

    /**
     * Returns the fallback currency-to-point conversion-multiplier for products, used by product import.
     */
    public function getProductFallbackRate(Mage_Core_Model_Store $store, Mage_Catalog_Model_Product $product): ?float
    {
        $helper = Mage::helper("awardit_points");

        $val = $helper->getRate($store);

        return $val > 0 ? $val : null;
    }

    /**
     * Returns the approximate amount of points available to the customer to be
     * used for the display of available points, null indicates the customer is
     * not eligible for this points currency.
     *
     * NOTE: Does not need to validate the actual amount is present, this
     * method is just to indicate the available value to the customer.
     */
    public function getCustomerPointsBalance(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): int {
        return Awardit_Points_Model_PointsData::fromCustomer($customer)->active;
    }

    /**
     * Returns true if the customer points balance is used to pay for taxes,
     * if included tax will count towards the total points to deduct.
     *
     * NOTE: Default should be true.
     */
    public function getCustomerPointsBalanceIncludesTax(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): bool {
        // Customer points are already paid for with VAT
        return false;
    }

    /**
     * Return true if the supplied customer is allowed to redeem points of this
     * type in the supplied store.
     */
    public function getCustomerRedemptionAllowed(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): bool {
        return (bool)$customer->getAwarditMemberId();
    }

    /**
     * Updates the customer point balance from any remote points source and
     * returns the new balance.
     */
    public function getUpdatedCustomerPointsBalance(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): int {
        $helper = Mage::helper("awardit_points");
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
            throw new Awardit_Points_JavaSessionException();
        }

        Mage::getModel("awardit_points/customer")->updatePoints($store, $customer, $javaSession);

        return $this->getCustomerPointsBalance($store, $customer);
    }

    /**
     * Perform extra validation or preparation before the actual quote payment
     * is initiated.
     *
     * @param Mage_Sales_Model_Order $order
     * @param Mage_Sales_Model_Quote $quote
     */
    public function onQuoteSubmitBefore(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void {
        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        $javaSession = $helper->getJavaSession();
        $store = $quote->getStore();
        $customer = $quote->getCustomer();

        if (! $javaSession) {
            throw new Awardit_Points_JavaSessionException();
        }

        if (! $customer->getAwarditMemberId()) {
            Mage::throwException(sprintf("Missing Awardit customer id for quote %d", $quote->getId() ?: ''));
        }

        $quote->setAwarditMemberId($customer->getAwarditMemberId());
        $order->setAwarditMemberId($customer->getAwarditMemberId());

        // Check for products with awardit_point_source_restriction
        $foundRestriction = false;
        foreach ($quote->getAllItems() as $item) {
            // Only check "real" products
            if (
                !in_array($item->getProductType(), [
                    Mage_Catalog_Model_Product_Type::TYPE_SIMPLE,
                    Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL
                ])
            ) {
                continue;
            }

            $product = $item->getProduct();
            $product->load($product->getId() ?: ''); // Force load attributes

            $currentRestriction = $product->getAwarditRestrictionPid();
            if ($foundRestriction !== false && $currentRestriction !== $foundRestriction) {
                throw new Exception(sprintf(
                    "%s: Cart contains conflicting points-restricted products on order '%s' from quote %d",
                    __METHOD__,
                    $order->getIncrementId(),
                    $quote->getId() ?: ''
                ));
            }

            $foundRestriction = $currentRestriction;
        }

        $quote->setAwarditPointSourceRestriction($foundRestriction ?: null);
        $order->setAwarditPointSourceRestriction($foundRestriction ?: null);

        $quote->save();

        $billing = $order->getBillingAddress();

        if (! $billing->getId()) {
            throw new Exception(sprintf(
                "%s: Billing address missing on order '%s' from quote %d",
                __METHOD__,
                $order->getIncrementId(),
                $quote->getId() ?: ''
            ));
        }

        $api->updateCustomer($customer, $store, $javaSession, $billing);

        // We no longer synchronize the order products, they are only activated
        // after having been imported into the java monolith
        //
        // $api->addProductsFromOrder($store, $order);
    }

    /**
     * Perform payment or authorization of points, crashing here will abort the
     * order and cancel the already pending payment.
     *
     * This is triggered BEFORE payment is performed.
     *
     * NOTE: This is usually run inside of a transaction when placing an order,
     * all failures will rollback any database-writes to the `core_write` adapter.
     *
     * @param Mage_Sales_Model_Order $order
     */
    public function onOrderPlaceBefore(Mage_Sales_Model_Order $order): void
    {
        return;
    }

    /**
     * Perform payment of points, crashing here will abort the order and cancel
     * the already pending payment.
     *
     * This is triggered AFTER payment is performed.
     *
     * NOTE: This is usually run inside of a transaction when placing an order,
     * all failures will rollback any database-writes to the `core_write` adapter.
     *
     * @param Mage_Sales_Model_Order $order
     */
    public function onOrderPlaceEnd(Mage_Sales_Model_Order $order): void
    {
        $api = Mage::getSingleton("awardit_points/api");
        $helper = Mage::helper("awardit_points");
        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore($order->getStoreId());
        $quote = $order->getQuote();
        $javaSession = $helper->getJavaSession();

        if (! $javaSession) {
            try {
                $order->cancel()
                    ->setState(Mage_Sales_Model_Order::STATE_CANCELED, 'canceled', 'Payment Failed', false)
                    ->save();
            } catch (Exception $e) {
                Mage::logException($e);
            }

            throw new Exception(sprintf(
                "Missing JavaSession for customer %d and quote %s",
                $order->getAwarditMemberId(),
                $quote->getId() ?: ''
            ));
        }

        try {
            $response = $api->debitPoints($store, $javaSession, $order);
            $pointSources = $response['pointSources'] ?: null;
            $order->setAwarditPointSources(json_encode($pointSources));
        } catch (Exception $e) {
            $order->cancel()
                ->setState(Mage_Sales_Model_Order::STATE_CANCELED, 'canceled', 'Payment Failed', false)
                ->save();

            throw $e;
        }
    }

    /**
     * Rollback anything previously done, payment or similar failed.
     *
     * Database writes should already have been rolled back, but make sure.
     *
     * @param Mage_Sales_Model_Order $order
     * @param Mage_Sales_Model_Quote $quote
     */
    public function onQuoteSubmitFailure(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void {
        return;
    }

    /**
     * This is always run if we did not previously error out.
     *
     * @param Mage_Sales_Model_Order $order
     * @param Mage_Sales_Model_Quote $quote
     */
    public function onQuoteSubmitAfter(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void {
        $session = Mage::getSingleton("customer/session");
        $helper = Mage::helper("awardit_points");
        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore();
        $customer = $session->getCustomer();
        $javaSession = $helper->getJavaSession();

        if ($javaSession) {
            try {
                Mage::getModel("awardit_points/customer")->updatePoints($store, $customer, $javaSession);
            } catch (Exception $e) {
                Mage::logException($e);
            }
        } else {
            Mage::log(sprintf(
                "%s: Missing Java Session for customer (id:%d,email:'%s')",
                __METHOD__,
                $customer->getId() ?: '',
                $customer->getEmail()
            ));
        }
    }
}
