<?php

class Awardit_Points_Model_Sales_Quote_Address_Total_Awardit extends Mage_Sales_Model_Quote_Address_Total_Abstract
{
    public const MIN_RATE = 0.01;

    protected $_code = "awardit";

    public function collect(Mage_Sales_Model_Quote_Address $address)
    {
        parent::collect($address);

        if (! count($this->_getAddressItems($address))) {
            $address->setTotalAmount($this->getCode(), 0);
            $address->setBaseTotalAmount($this->getCode(), 0);
            $address->setAwarditAmountTax(0);
            $address->setBaseAwarditAmountTax(0);

            return $this;
        }

        // TODO: Abort here if we are not virtual and this is billing?

        $helper = Mage::helper("awardit_points");
        $quote = $address->getQuote();
        $store = $quote->getStore();
        $customer = $quote->getCustomer();

        if (! $helper->isEnabled($store) || ! $quote->getAwarditPoints()) {
            $address->setTotalAmount($this->getCode(), 0);
            $address->setBaseTotalAmount($this->getCode(), 0);
            $address->setAwarditAmountTax(0);
            $address->setBaseAwarditAmountTax(0);

            return $this;
        }

        $rate = (float)($quote->getAwarditRate() ?: $helper->getRate($store));
        $ownedPoints = $customer ? (int)$customer->getAwarditPoints() : 0;

        if ($rate < self::MIN_RATE) {
            Mage::throwException(sprintf("%s: Invalid rate on quote %d", __METHOD__, $quote->getId()));
        }

        $points = min((int)$quote->getAwarditPoints(), $ownedPoints);
        $totalValue = $points / $rate;

        $orderVat = $this->getOrderTotalVat($address->getQuote());
        $orderExVat = $this->getOrderTotalExVat($address->getQuote());

        $baseExVatAmount = min($totalValue, $orderExVat);

        // Correct if we try to pay for more than the full amount
        $newPoints = min(ceil(abs($baseExVatAmount * $rate)), $ownedPoints);

        $baseTaxAmount = $orderVat * $baseExVatAmount / $orderExVat;
        $baseAmount = $baseExVatAmount + $baseTaxAmount;

        $amount = $store->convertPrice($baseAmount, false);
        $taxAmount = $store->convertPrice($baseTaxAmount, false);

        $address->setTotalAmount($this->getCode(), -$amount);
        $address->setBaseTotalAmount($this->getCode(), -$baseAmount);
        $address->setAwarditAmountTax($taxAmount);
        $address->setBaseAwarditAmountTax($baseTaxAmount);

        $quote->setAwarditPoints($newPoints);
        $quote->setAwarditRate($rate);

        return $this;
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address)
    {
        $address->addTotal([
            "code"           => $this->getCode(),
            "value"          => $address->getAwarditAmount(),
            "value_incl_tax" => $address->getAwarditAmount(),
            "value_excl_tax" => $address->getAwarditAmount() - $address->getAwarditAmountTax(),
        ]);

        return $this;
    }

    public function getOrderTotalExVat(Mage_Sales_Model_Quote $quote): float
    {
        $store = $quote->getStore();
        $baseTotals = 0;

        foreach ($quote->getAddressesCollection() as $a) {
            $baseTotals += array_sum(array_filter($a->getAllBaseTotalAmounts(), function ($k) {
                return strstr($k, "tax") === false;
            }, ARRAY_FILTER_USE_KEY));
        }

        return $store->roundPrice($baseTotals);
    }

    public function getOrderTotalVat(Mage_Sales_Model_Quote $quote): float
    {
        $store = $quote->getStore();
        $baseTotals = 0;

        foreach ($quote->getAddressesCollection() as $a) {
            $baseTotals += array_sum(array_filter($a->getAllBaseTotalAmounts(), function ($k) {
                return strstr($k, "tax") !== false;
            }, ARRAY_FILTER_USE_KEY));
        }

        return $store->roundPrice($baseTotals);
    }
}
