<?php

/**
 * @psalm-import-type ApiPointsData from Awardit_Points_Model_Api
 */
class Awardit_Points_Model_PointsData
{
    /**
     * Points which can be redeemed.
     *
     * @readonly
     * @var int
     */
    public $active;
    /**
     * Exact amount of points which can be redeemed.
     *
     * @readonly
     * @var float
     */
    public $activeExact;
    /**
     * Ranking points awarded to the member.
     *
     * @readonly
     * @var int
     */
    public $rank;
    /**
     * Incoming points from affiliates, not yet usable.
     *
     * @readonly
     * @var int
     */
    public $pending;
    /**
     * Amount of points available to the group.
     *
     * @readonly
     * @var int
     */
    public $group;
    /**
     * Exact amount of points earned by the group.
     *
     * @readonly
     * @var float
     */
    public $groupExact;
    /**
     * Ranking points earned by the group the member is in.
     *
     * @readonly
     * @var int
     */
    public $groupRank;
    /**
     * Amount of points earned by the member on their own.
     *
     * @readonly
     * @var int
     */
    public $own;
    /**
     * Exact amount of points earned by the member on their own.
     *
     * @readonly
     * @var float
     */
    public $ownExact;
    /**
     * Ranking points earned by the member on their own.
     *
     * @readonly
     * @var int
     */
    public $ownRank;
    /**
     * Amount of points expiring in the next period.
     *
     * @readonly
     * @var int
     */
    public $expiring;
    /**
     * When the next expiration period ends, and expiring points will be expired.
     *
     * @readonly
     * @var string
     */
    public $expiredate;

    /**
     * Creates a new points object from the available JSON data in the customer object.
     *
     * @param Mage_Customer_Model_Customer $customer
     */
    public static function fromCustomer(Mage_Customer_Model_Customer $customer): self
    {
        $json = $customer->getAwarditPointsData();
        /**
         * @var ApiPointsData
         */
        $data = ($json ? json_decode($json, true) : null) ?? [];

        return new self($data);
    }

    /**
     * @param ApiPointsData $pointsData
     */
    public function __construct(array $pointsData)
    {
        $this->active = self::filterPoints(
            isset($pointsData["exactpoints"])
                ? floor($pointsData["exactpoints"])
                : ""
        ); // Use exactpoints
        $this->activeExact = self::filterExact($pointsData["exactpoints"] ?? "");
        $this->rank = self::filterPoints($pointsData["rankpoints"] ?? "");
        $this->pending = self::filterPoints($pointsData["pendingpoints"] ?? "");

        $this->group = self::filterPoints($pointsData["grouppoints"] ?? "");
        $this->groupExact = self::filterExact($pointsData["exactgrouppoints"] ?? "");
        $this->groupRank = self::filterPoints($pointsData["grouprankpoints"] ?? "");

        $this->own = self::filterPoints($pointsData["ownpoints"] ?? "");
        $this->ownExact = self::filterExact($pointsData["exactownpoints"] ?? "");
        $this->ownRank = self::filterPoints($pointsData["ownrankpoints"] ?? "");

        $this->expiring = self::filterPoints($pointsData["expirepoints"] ?? "");
        $this->expiredate = (string)($pointsData["expiredate"] ?? "");

        $this->waitingPoints = self::filterPoints($pointsData["waitingPoints"] ?? "");
        $this->minThreshold = self::filterPoints($pointsData["minThreshold"] ?? "");
        $this->earnedPoints = self::filterPoints($pointsData["earnedPoints"] ?? "");
    }

    private static function filterPoints(string $pointsData): int
    {
        return (int)preg_replace("/[^-0-9]/", "", $pointsData);
    }

    private static function filterExact(string $pointsData): float
    {
        return (float)preg_replace("/[^-0-9\.]/", "", $pointsData);
    }
}
