<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class Awardit_Points_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        $store = Mage::app()->getStore();

        if( ! Mage::helper("awardit_points")->isEnabled($store)) {
            return null;
        }

        switch($typeName) {
        case "ActivateAwarditAffiliateResult":
            return $this->object("The result of an activateAwarditAffiliate mutation");

        case "ActivateAwarditAffiliateResultType":
            return $this->enum("The result type of an activateAwarditAffiliate mutation", [
                Awardit_Points_Model_Customer_Affiliate::SUCCESS => [
                    "description" => "The affiliate was activated successfully",
                ],
                Awardit_Points_Model_Customer_Affiliate::NOT_ACTIVATED => [
                    "description" => "The affiliate was not activated",
                ],
                Awardit_Points_Model_Customer_Affiliate::ERROR_NOT_LOGGED_IN => [
                    "description" => "The customer is not logged in",
                ],
            ]);

        case "AwarditAgreement":
            return $this->object("Agreement information for the current partner and possibly logged in user")
                    ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "AwarditMemberInfo":
            return $this->object("Information about an awardit member")
                    ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "AwarditStoreInfo":
            return $this->object("Information about awardit partner");

        case "ChangeAwarditPasswordResult":
            return $this->object("The result of a changeAwarditPassword mutation");

        case "ChangeAwarditPasswordResultType":
            return $this->enum("The result type of a changeAwarditPassword mutation", [
                Awardit_Points_Model_Customer::SUCCESS => [
                    "description" => "Successfully updated password",
                ],
                Awardit_Points_Model_Customer::ERROR_NOT_LOGGED_IN => [
                    "description" => "User is not logged in",
                ],
                Awardit_Points_Model_Customer::ERROR => [
                    "description" => "Something went wrong",
                ],
            ]);

        case "LoginAwarditCustomerResult":
            return $this->object("The result of a loginAwarditCustomer mutation");

        case "LoginAwarditCustomerResultType":
            return $this->enum("The result type of a loginAwarditCustomer mutation", [
                Awardit_Points_Model_Customer::SUCCESS => [
                    "description" => "The customer was logged in successfully",
                ],
                Awardit_Points_Model_Customer::ERROR_LOGIN_FAILED => [
                    "description" => "The attempted login failed",
                ],
                Awardit_Points_Model_Customer::ERROR_MISSING_CUSTOMER => [
                    "description" => "The customer could not be found at Awardit.",
                ],
                Awardit_Points_Model_Customer::EMAILVERIFY_TYPE_REQUEST => [
                    "description" => "A verification code needs to be entered by the customer",
                ],
                Awardit_Points_Model_Customer::EMAILVERIFY_TYPE_SENT => [
                    "description" => "A verification email has been sent to the customer",
                ],
                Awardit_Points_Model_Customer::EMAILVERIFY_TWOFACTOR => [
                    "description" => "2FA verification code needs to be entered by the customer",
                ],
            ]);

        case "ResetAwarditCustomerPasswordResult":
            return $this->object("The result of a resetAwarditPassword mutation");

        case "ResetAwarditCustomerPasswordResultType":
            return $this->enum("The result type of a resetAwarditPassword mutation", [
                Awardit_Points_Model_Customer::SUCCESS => [
                    "description" => "Successfully sent a reset password email",
                ],
                Awardit_Points_Model_Customer::ERROR => [
                    "description" => "Something went wrong, check that the email is spelled correctly",
                ],
            ]);

        case "AwarditSetCustomerMailnotifyResult":
            return $this->object("The result of a loginAwarditCustomer mutation");

        case "AwarditSetCustomerMailnotifyResultType":
            return $this->enum("The result type of a awarditSetCustomerMailnotify mutation", [
                Awardit_Points_Model_Customer::SUCCESS => [
                    "description" => "Successfully sent a reset password email",
                ],
                Awardit_Points_Model_Customer::ERROR => [
                    "description" => "Something went wrong, check that the email is spelled correctly",
                ],
                Awardit_Points_Model_Customer::ERROR_NOT_LOGGED_IN => [
                    "description" => "User is not logged in",
                ],
            ]);

        case "AwarditAgreementAgreeResult":
            return $this->enum("The result type of a awarditAgreeAgreement mutation", [
                Awardit_Points_Model_Customer::SUCCESS => [
                    "description" => "Successfully agreed to the agreement",
                ],
                Awardit_Points_Model_Customer::NOT_MODIFIED => [
                    "description" => "Customer has previously agreed",
                ],
                Awardit_Points_Model_Customer::ERROR_NOT_LOGGED_IN => [
                    "description" => "User is not logged in",
                ],
            ]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        $store = Mage::app()->getStore();

        if( ! Mage::helper("awardit_points")->isEnabled($store)) {
            return [];
        }

        switch($typeName) {
        case "ActivateAwarditAffiliateResult":
            return [
                "result" => $this->field("ActivateAwarditAffiliateResultType!", "Type of result from activateAwarditAffiliate mutation")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "AwarditAgreement":
            return [
                "content" => $this->field("String!", "HTML content, filtered"),
            ];

        case "AwarditMemberInfo":
            return [
                "memberId" => $this->field("String", "Awardit member id of the customer, if no id is present then the customer has not logged in properly and should not be able to use the points payment or related queries."),
                "activePoints" => $this->field("Int!", "Amount of awardit points available for purchase"),
                "expiringPoints" => $this->field("Int!", "Number of points which are soon expiring"),
                "expiringPointsDate" => $this->field("String", "Date when points are expiring"),
                "rank" => $this->field("Int!", "Number of ranking points the member has earned, more = better"),
                "pendingPoints" => $this->field("Int!", "Incoming points from affiliates, not yet usable"),
                "groupPoints" => $this->field("Int!", "Amount of points available to / earned by the group"),
                "groupRank"  => $this->field("Int!", "Number of ranking points the group has earned"),
                "ownPoints" => $this->field("Int!", "Amount of points earned by the member on their own. own <= active <= group."),
                "ownRank" => $this->field("Int!", "Ranking points earned by the member on their own"),

                "waitingPoints" => $this->field("Int!", "Waiting points that can be transformed to available points"),
                "minThreshold" => $this->field("Int!", "Threshold to unlock store"),
                "earnedPoints" => $this->field("Int!", "Points earned in current period"),

                "rate" => $this->field("Float!", "Number of points per unit of currency"),

                "numberOfEmployees" => $this->field("Int!", "Number of employees in the organization"),
                "mailnotify" => $this->field("Boolean!", "If notification emails are to be sent to the customer"),
                "userType" => $this->field("Int", "Customer user type if any"),
                "partneruserid" => $this->field("Int!", "Partneruserid for the customer"),
                "agreementAccepted" => $this->field("Boolean!", "If the member has accepted the agreement"),
            ];

        case "AwarditStoreInfo":
            return [
                "partneruserid" => $this->field("Int!", "Default partneruserid for store partner, use customer.awardit.partneruserid if logged in")
                    ->setResolver(
                    /**
                     * @param mixed $unusedSrc
                     */
                    function($unusedSrc, array $unusedArgs, MageQL_Core_Model_Context $ctx): int {
                        $partneruserid = Mage::helper("awardit_points")->getPartnerId($ctx->getStore());

                        return (int)$partneruserid;
                    }),
            ];

        case "ChangeAwarditPasswordResult":
            return [
                "result" => $this->field("ChangeAwarditPasswordResultType!", "Type of result from changeAwarditPassword mutation")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "LoginAwarditCustomerResult":
            return [
                "result" => $this->field("LoginAwarditCustomerResultType!", "The result"),
                "JSESSIONID" => $this->field("String", "Awardit Java Session id"),
                "customer" => $this->field("Customer", "The logged in customer if successful"),
                "memberuserid" => $this->field("String", "Memberuserid for the logged in customer"),
                "mailto" => $this->field("String", "The email address the verification email is sent to, if sent or to be sent"),
                "twofactorSkipDays" => $this->field("Int", "Amount of days to skip twofactor verification"),
            ];

        case "Customer":
            return [
                "awardit" => $this->field("AwarditMemberInfo!", "Information related to awardit points")
                    ->setResolver("Awardit_Points_Model_Customer::resolveAwardit"),
            ];

        case "Mutation":
            return [
                "activateAwarditAffiliate" => $this->field("ActivateAwarditAffiliateResult!", "Attempts to activate an awardit affiliate for a customer")
                    ->addArgument("id", $this->argument("ID!", "Awardit affiliate "))
                    ->setResolver("Awardit_Points_Model_Customer_Affiliate::mutateActivate"),
                "loginAwarditCustomer" => $this->field("LoginAwarditCustomerResult!", "Attempts to login a customer using awardit redirect data")
                    ->addArgument("username", $this->argument("String!", "Username for awardit account"))
                    ->addArgument("password", $this->argument("String!", "Password for awardit account"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateLogin"),
                "loginAwarditCustomerOtp" => $this->field("LoginAwarditCustomerResult!", "When system requires 2FA, send in OTP code")
                    ->addArgument("code", $this->argument("String!", "OTP code received by email"))
                    ->addArgument("remember", $this->argument("Boolean!", "Remember device"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateLoginOtp"),
                "loginAsAwarditCustomer" => $this->field("LoginAwarditCustomerResult!", "Attempts to login as an awardit customer from admin")
                    ->addArgument("loginkey", $this->argument("String!", "Login key"))
                    ->addArgument("hiddenlogin", $this->argument("String!", "Hiddenlogin key"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateLoginAsCustomer"),
                "resetAwarditPassword" => $this->field("ResetAwarditCustomerPasswordResult!", "Resets the awardit customer password by sending an email with a new password")
                    ->addArgument("email", $this->argument("String!", "Email for awardit account"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateResetPassword"),
                "changeAwarditPassword" => $this->field("ChangeAwarditPasswordResult!", "Attempts to change the password for the currently logged in awardit customer")
                    ->addArgument("oldPassword", $this->argument("String!", "Old password"))
                    ->addArgument("newPassword", $this->argument("String!", "New password"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateChangePassword"),
                "awarditSetCustomerMailnotify" => $this->field("AwarditSetCustomerMailnotifyResult!", "Update email preferences")
                    ->addArgument("value", $this->argument("Boolean!", "If to send emails"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateSetMailnotify"),
                "awarditAgreementAgree" => $this->field("AwarditAgreementAgreeResult!", "Agree to the agreement")
                    ->setResolver("Awardit_Points_Model_Customer::mutateAgreementAgree"),
                "awarditRequestAccountVerificationEmail" => $this->field("Boolean!", "Request an email to verify the account, use when the loginAwarditCustomer returns requestVerification")
                    ->addArgument("memberuserid", $this->argument("String!", "The memberuserid to send a verification email to"))
                    ->setResolver("Awardit_Points_Model_Customer::mutateRequestAccountVerificationEmail"),
            ];

        case "Query":
            return [
                "awarditAgreement" => $this->field("AwarditAgreement", "Awardit agreement for the partner and possibly logged in user")
                    ->setResolver("Awardit_Points_Model_Customer::resolveAgreement"),
            ];

        case "ResetAwarditCustomerPasswordResult":
            return [
                "result" => $this->field("ResetAwarditCustomerPasswordResultType!", "Type of result")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "StoreInfo":
            return [
                "awardit" => $this->field("AwarditStoreInfo!", "Store information for awardit partner")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "AwarditSetCustomerMailnotifyResult":
            return [
                "result" => $this->field("AwarditSetCustomerMailnotifyResultType!", "Type of result")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        default:
            return [];
        }
    }
}
