<?php

declare(strict_types=1);

use OpenSearch\Common\Exceptions\Missing404Exception;
use Psr\Log\LoggerInterface;

class Awardit_Opensearch_Model_Resource_Product
{
    /** @var Awardit_Opensearch_Helper_Data */
    protected $helper;
    protected LoggerInterface $logger;

    public function __construct()
    {
        $this->helper = Mage::helper('awardit_opensearch');
        $this->logger = $this->helper->getPsrLogger();
    }

    public function reindexAll(): void
    {
        $this->indexProducts();
    }

    /**
     * @param  Array<int>|null $productIds
     * @return void
     */
    public function indexProducts($productIds = null): void
    {
        foreach ($this->helper->getStores() as $store) {
            $products = Mage::getResourceModel('catalog/product_collection');
            $products->setStore($store);
            $products->setVisibility(Mage::getSingleton('catalog/product_visibility')->getVisibleInSearchIds());
            $products->addAttributeToFilter('sku', ['neq' => '']); // Required in feed,

            if ($productIds !== null && !empty($productIds)) {
                $products->addFieldToFilter('entity_id', ['in' => $productIds]);
            }

            $indexedIds = $products->getAllIds();

            foreach ($indexedIds as $productId) {
                $this->indexProduct($store, (int) $productId);
            }

            if ($productIds !== null && !empty($productIds)) {
                $toDelete = array_diff($productIds, $indexedIds);
                foreach ($toDelete as $productId) {
                    $this->deleteProduct($store, (int) $productId);
                }
            }
        }
    }

    /**
     * @param Mage_Core_Model_Store $store
     * @param int $id
     * @return void
     */
    private function indexProduct(Mage_Core_Model_Store $store, int $id): void
    {
        $product = Mage::getModel('catalog/product')->setStoreId($store->getId())->load($id);
        $params = [
            'index' => $store->getCode(),
            'id' => $product->getId(),
            'body'  => [
                'sku' => $product->getSku(),
                'entity_id' => $product->getId(),
                'name' => $product->getName(),
                'price' => $product->getPrice(),
                'description' => $product->getDescription(),
                'short_description' => $product->getShortDescription(),
                'manufacturer' => $product->getAttributeText('manufacturer') ?: null,
            ],
        ];
        $this->logger->info("Indexing in Elastic: " . json_encode($params));
        $response = $this->_call('index', $params);
        $this->logger->info("Response from Elastic: " . json_encode($response));
    }

    /**
     * @param  Array<int>|null $productIds
     * @return void
     */
    public function deleteProducts($productIds = null): void
    {
        if ($productIds === null) {
            return;
        }

        foreach ($this->helper->getStores() as $store) {
            foreach ($productIds as $productId) {
                $this->deleteProduct($store, (int) $productId);
            }
        }
    }

    /**
     * @param Mage_Core_Model_Store $store
     * @param int $id
     * @return void
     */
    private function deleteProduct(Mage_Core_Model_Store $store, int $id): void
    {
        $params = [
            'index' => $store->getCode(),
            'id' => $id,
        ];
        $this->logger->info("Indexing in Elastic: " . json_encode($params));
        $response = $this->_call('delete', $params);
        $this->logger->info("Response from Elastic: " . json_encode($response));
    }

    /**
     * @param string $method
     * @param mixed ...$args
     * @return mixed
     */
    private function _call(string $method, ...$args)
    {
        $client = $this->helper->getClient();
        try {
            return call_user_func_array([$client, $method], $args);
        } catch (Missing404Exception $e) {
            // @todo: do nothing
        } catch (\Exception $e) {
            $this->logger->error("Error from Elastic: {$e->getMessage()}");
        }
    }
}
