<?php

declare(strict_types=1);

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Awardit_Opensearch_Model_Indexer_Opensearch extends Mage_Index_Model_Indexer_Abstract
{
    public const UPDATE_KEY                  = "Awardit_Opensearch_update_product_id";
    public const DELETE_KEY                  = "Awardit_Opensearch_delete_product_id";
    public const MASS_ACTION_PRODUCT_IDS_KEY = "Awardit_Opensearch_mass_action_product_ids";
    public const MASS_ACTION_TYPE_KEY        = "Awardit_Opensearch_mass_action_type";

    protected $_matchedEntities = [
        Mage_CatalogInventory_Model_Stock_Item::ENTITY => array(
            Mage_Index_Model_Event::TYPE_SAVE
        ),
        Mage_Catalog_Model_Product::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_DELETE,
            Mage_Index_Model_Event::TYPE_MASS_ACTION,
        ],
        //// We need all the ones below since they affect the indexed data
        Mage_Catalog_Model_Category::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE
        ],
        Mage_Core_Model_Config_Data::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE
        ],
        Mage_Catalog_Model_Resource_Eav_Attribute::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_DELETE,
        ],
    ];

    /**
     * @return string
     */
    public function getName(): string
    {
        return "Opensearch";
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return "Opensearch";
    }

    protected function _registerEvent(Mage_Index_Model_Event $event): void
    {
        switch ($event->getEntity()) {
            case Mage_Catalog_Model_Product::ENTITY:
                $this->registerProduct($event);
                break;
            case Mage_Catalog_Model_Resource_Eav_Attribute::ENTITY:
                $this->registerAttribute($event);
                break;
            case Mage_Core_Model_Config_Data::ENTITY:
                $this->registerConfig($event);
                break;
            case Mage_Catalog_Model_Category::ENTITY:
                $this->registerCategory($event);
                break;
            case Mage_CatalogInventory_Model_Stock_Item::ENTITY:
                $this->registerStockItem($event);
                break;
        }
    }

    protected function registerAttribute(Mage_Index_Model_Event $event): void
    {
        $attr = $event->getDataObject();

        if (
            $attr->dataHasChangedFor("is_filterable_in_search") ||
            $attr->dataHasChangedFor("is_visible_in_advanced_search") ||
            $attr->dataHasChangedFor("is_searchable") ||
            $attr->dataHasChangedFor("frontend_input") ||
            $attr->dataHasChangedFor("backend_type") ||
            // is_html:
            $attr->dataHasChangedFor("is_html_allowed_on_front") ||
            $attr->dataHasChangedFor("is_wysiwyg_enabled")
        ) {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerCategory(Mage_Index_Model_Event $event): void
    {
        if ($event->getType() !== Mage_Index_Model_Event::TYPE_SAVE) {
            return;
        }

        $category   = $event->getDataObject();
        $productIds = $category->getAffectedProductIds();

        if ($productIds) {
            $event->addNewData(self::MASS_ACTION_PRODUCT_IDS_KEY, $productIds);
            $event->addNewData(self::MASS_ACTION_TYPE_KEY, "save");
        } else {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerConfig(Mage_Index_Model_Event $event): void
    {
        if (stripos($event->getDataObject()->getPath(), "awardit_opensearch") === 0) {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerProduct(Mage_Index_Model_Event $event): void
    {
        $dataObject = $event->getDataObject();

        switch ($event->getType()) {
            case Mage_Index_Model_Event::TYPE_SAVE:
                $event->addNewData(self::UPDATE_KEY, $dataObject->getEntityId());
                break;
            case Mage_Index_Model_Event::TYPE_DELETE:
                $event->addNewData(self::DELETE_KEY, $dataObject->getEntityId());
                break;
            case Mage_Index_Model_Event::TYPE_MASS_ACTION:
                $event->addNewData(self::MASS_ACTION_PRODUCT_IDS_KEY, $dataObject->getProductIds());
                $event->addNewData(self::MASS_ACTION_TYPE_KEY, $dataObject->getActionType());
                break;
        }
    }

    protected function registerStockItem(Mage_Index_Model_Event $event): void
    {
        if ($event->getType() !== Mage_Index_Model_Event::TYPE_SAVE) {
            return;
        }

        $event->addNewData(self::UPDATE_KEY, $event->getDataObject()->getProductId());
    }

    protected function _processEvent(Mage_Index_Model_Event $event): void
    {
        $dataObject = $event->getNewData();
        $resource   = Mage::getResourceModel("awardit_opensearch/product");

        if (!empty($dataObject[self::UPDATE_KEY])) {
            $resource->indexProducts([$dataObject[self::UPDATE_KEY]]);
        }

        if (!empty($dataObject[self::DELETE_KEY])) {
            $resource->deleteProducts([$dataObject[self::DELETE_KEY]]);
        }

        if (!empty($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY])) {
            switch ($dataObject[self::MASS_ACTION_TYPE_KEY]) {
                case "remove":
                    $resource->deleteProducts($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY]);
                    break;
                default:
                    $resource->indexProducts($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY]);
            }
        }
    }

    /**
     * Rebuild all index data
     *
     * @return void
     */
    public function reindexAll(): void
    {
        Mage::getResourceModel("awardit_opensearch/product")->reindexAll();
    }
}
