<?php

declare(strict_types=1);

use Awardit\MagentoPsr\Psr3\Logger;
use OpenSearch\Client;
use OpenSearch\ClientBuilder;
use Aws\Credentials\Credentials;
use Aws\Credentials\CredentialProvider;

class Awardit_Opensearch_Helper_Data
{
    private const LOG_CHANNEL = 'opensearch';
    public const CONFIG_ENABLED = "awardit_opensearch/opensearch/enabled";
    public const CONFIG_HOST = "awardit_opensearch/opensearch/host";
    public const CONFIG_USERNAME = "awardit_opensearch/opensearch/username";
    public const CONFIG_PASSWORD = "awardit_opensearch/opensearch/password";
    public const CONFIG_USE_AWS = "awardit_opensearch/opensearch/use_aws";
    public const CONFIG_AWS_KEY = "awardit_opensearch/opensearch/aws_key";
    public const CONFIG_AWS_SECRET = "awardit_opensearch/opensearch/aws_secret";
    public const CONFIG_AWS_REGION = "awardit_opensearch/opensearch/aws_region";
    public const CONFIG_SEARCH_MODE = "awardit_opensearch/opensearch/search_mode";

    private Logger $psrLogger;

    public function __construct()
    {
        $this->psrLogger = new Logger(self::LOG_CHANNEL);
    }
    /**
     * @return Array<Mage_Core_Model_Store>
     */
    public function getStores(): array
    {
        // Only return stores that have Opensearch enabled
        return array_filter(Mage::app()->getStores(), function ($store) {
            return (bool) $store->getConfig(self::CONFIG_ENABLED) ?: false;
        });
    }

    /**
     * @param Mage_Core_Model_Store|null $store
     * @return Client
     */
    public function getClient(Mage_Core_Model_Store $store = null, array $connectionParams = []): Client
    {
        $host = Mage::getStoreConfig(self::CONFIG_HOST, $store);
        $useAws = (bool) Mage::getStoreConfig(self::CONFIG_USE_AWS, $store);

        if (empty($connectionParams)) {
            $connectionParams = [
                'client' => [
                    'curl' => [
                        CURLOPT_CONNECTTIMEOUT => 3,
                    ]
                ]
            ];
        }
        $clientBuilder = ClientBuilder::create()
            ->setHosts([$host])
            ->setConnectionParams($connectionParams);
        // Check if AWS authentication should be used
        if ($useAws) {
            // AWS SigV4 authentication
            $awsKey = Mage::getStoreConfig(self::CONFIG_AWS_KEY, $store);
            $awsSecret = Mage::helper('core')->decrypt(Mage::getStoreConfig(self::CONFIG_AWS_SECRET, $store));
            $region = Mage::getStoreConfig(self::CONFIG_AWS_REGION, $store);

            // Create credential provider that returns a promise
            $credentialProvider = CredentialProvider::fromCredentials(
                new Credentials($awsKey, $awsSecret)
            );
            // Set AWS SigV4 authentication
            $clientBuilder->setSigV4Region($region)
                ->setSigV4Service('es')
                ->setSigV4CredentialProvider($credentialProvider);
        } else {
            // Basic authentication (existing behavior)
            $username = Mage::getStoreConfig(self::CONFIG_USERNAME, $store);
            $password = Mage::getStoreConfig(self::CONFIG_PASSWORD, $store);
            $clientBuilder->setBasicAuthentication($username, $password);
        }

        return $clientBuilder->build();
    }

    public function getPsrLogger(): Logger
    {
        return $this->psrLogger;
    }
}
