<?php

declare(strict_types=1);

use OpenSearch\Client;

class Awardit_Opensearch_Model_Catalog_Opensearch
{
    /** @var Client */
    private $client;
    /** @var Mage_Core_Model_Store */
    private $store;

    public function __construct(Mage_Core_Model_Store $store)
    {
        $this->store = $store;
        $helper = Mage::helper('awardit_opensearch');
        $customCurlParams = [
            'client' => [
                'curl' => [
                    CURLOPT_CONNECTTIMEOUT => 3,
                ]
            ]
        ];
        $this->client = $helper->getClient($store, $customCurlParams);
    }

    public function search(string $term): array
    {
        $search = $this->searchRaw($term);
        $results = [];

        if (!isset($search['hits']['hits'])) {
            // No hits
            return $results;
        }

        foreach ($search['hits']['hits'] as $hit) {
            $results[$hit['_source']['entity_id']] = $hit['_score'];
        }
        return $results;
    }

    private function searchRaw(string $term): array
    {
        $searchMode = $this->store->getConfig(Awardit_Opensearch_Helper_Data::CONFIG_SEARCH_MODE) ?: 'exact';

        // Build query based on search mode
        if ($searchMode === 'contains') {
            // Contains mode - matches partial words
            $query = [
                'bool' => [
                    'should' => [
                        // Exact and phrase matches - highest priority
                        [
                            'multi_match' => [
                                'query' => $term,
                                'type' => 'phrase',
                                'fields' => ["sku^20", "name^20", "manufacturer^6"],
                                'boost' => 2,
                            ],
                        ],
                        // Word matches
                        [
                            'multi_match' => [
                                'query' => $term,
                                'type' => 'best_fields',
                                'fields' => ["sku^10", "name^10", "manufacturer^3", "short_description", "description"],
                            ],
                        ],
                        // Prefix matching with wildcards for partial word matches
                        [
                            'query_string' => [
                                'query' => '*' . $term . '*',
                                'fields' => ["sku^5", "name^5", "manufacturer^2", "short_description^0.5", "description^0.5"],
                                'default_operator' => 'AND',
                            ],
                        ],
                    ],
                    'minimum_should_match' => 1,
                ],
            ];
        } else {
            // Exact mode - matches only complete words (default behavior)
            $query = [
                'multi_match' => [
                    'query' => $term,
                    'type' => 'best_fields',
                    'fields' => ["sku^10", "name^10", "manufacturer^3", "short_description", "description"],
                ],
            ];
        }

        $params = [
            'index' => $this->store->getCode(),
            'body' => [
                '_source' => ['entity_id'],
                'size' => 320,
                'query' => $query,
            ],
        ];

        $result = $this->client->search($params);
        return $result;
    }
}
