<?php

declare(strict_types=1);

namespace Awardit\MicrosoftTeamsWebhook\Test;

use Awardit\MicrosoftTeamsWebhook\Card\AdaptiveCard;
use Awardit\MicrosoftTeamsWebhook\Message;
use Awardit\MicrosoftTeamsWebhook\Test\Fixture\TestAdaptiveCard;
use Awardit\MicrosoftTeamsWebhook\Webhook;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;

final class WebhookTest extends TestCase {
    public function testErrorStatusCodes(): void {
        // Create a mock and queue two responses.
        $mock = new MockHandler([
            new Response(400, ['X-Foo' => 'Bar'], 'Hello, World'),
            new Response(500, ['X-Foo' => 'Bar'], 'Hello, World'),
        ]);
        $handlerStack = HandlerStack::create($mock);
        $client = new Client(['handler' => $handlerStack]);

        $webhook = new Webhook('https://google.com', $client);
        $message = new Message(new AdaptiveCard());
        $this->expectException(ClientException::class);
        $this->assertEquals('Hello, World', $webhook->send($message));
        $this->expectException(ServerException::class);
        $this->assertEquals('Hello, World', $webhook->send($message));
    }

    public function testStatusCodes(): void {
        // Create a mock and queue two responses.
        $mock = new MockHandler([
            new Response(200, ['X-Foo' => 'Bar'], 'Hello, World'),
        ]);
        $container = [];
        $history = Middleware::history($container);
        $handlerStack = HandlerStack::create($mock);
        $handlerStack->push($history);

        $client = new Client(['handler' => $handlerStack]);

        $webhook = new Webhook('https://google.com', $client);
        $message = new Message(new TestAdaptiveCard([
            'bodyText' => 'This is an example text block',
        ]));
        $webhook->send($message);

        foreach($container as $transaction) {
            $requestBody = (string)$transaction['request']->getBody();
            $expected = '{"type":"message","attachments":[{"contentType":"application\/vnd.microsoft.card.adaptive","contentUrl":null,"content":{"type":"AdaptiveCard","$schema":"http:\/\/adaptivecards.io\/schemas\/adaptive-card.json","version":"1.5","body":[{"type":"TextBlock","text":"This is an example text block","wrap":true,"isVisible":true,"isSubtle":false,"style":"default"}]}}]}';
            $this->assertEquals($expected, $requestBody);
        }
    }
}
