<?php

declare(strict_types=1);

namespace Awardit\Microauth\Test;

use Awardit\Microauth\{
    AccessToken,
    AuthenticatorInterface,
    Exception,
    JwtAuthenticator,
    TokenInterface,
    UnauthorizedException,
};
use Firebase\JWT\{
    JWT,
    Key,
};
use PHPUnit\Framework\TestCase;

/** @psalm-suppress UnusedClass */
class JwtAuthenticatorTest extends TestCase
{
    private static string $privateKey;
    private static string $publicKey;

    public static function setUpBeforeClass(): void
    {
        self::$privateKey = file_get_contents(__DIR__ . '/Fixtures/privateKey.txt');
        self::$publicKey = file_get_contents(__DIR__ . '/Fixtures/publicKey.txt');
    }

    public function testJwtAuthenticator(): void
    {
        $jwtAuthenticator = new JwtAuthenticator([
            'myKid' => new Key(self::$publicKey, 'RS256'),
        ], ['example.com'], 'example.com');
        $this->assertInstanceOf(AuthenticatorInterface::class, $jwtAuthenticator);
        $jwt = JWT::encode([
            'aud' => 'example.com',
            'iss' => 'example.com',
            'sub' => 'user@example.com',
            'jti' => 'jti',
            'sid' => 'sid',
            'client_id' => 'client_id',
            'iat' => time(),
            'exp' => time() + 1000,
        ], self::$privateKey, 'RS256', 'myKid');
        $token = $jwtAuthenticator->authenticate($jwt);
        $this->assertInstanceOf(TokenInterface::class, $token);
        $this->assertInstanceOf(AccessToken::class, $token);
    }

    public function testEmptyKeys(): void
    {
        $jwtAuthenticator = new JwtAuthenticator([], [], 'audience');
        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Invalid Authenticator key-configuration: Key may not be empty");
        $jwtAuthenticator->authenticate('abc');
    }

    public function testInvalidKey(): void
    {
        $jwtAuthenticator = new JwtAuthenticator([
            'myKid' => new Key('myKid', 'RS256'),
        ], [], 'example.com');
        $this->expectExceptionMessage("Invalid JWT: Wrong number of segments");
        $jwtAuthenticator->authenticate('abc');
    }

    public function testInvalidIssClaim(): void
    {
        $jwtAuthenticator = new JwtAuthenticator([
            'myKid' => new Key(self::$publicKey, 'RS256'),
        ], [], 'example.com');
        $jwt = JWT::encode([
            'aud' => 'example.com',
            'iss' => 'example.com',
        ], self::$privateKey, 'RS256', 'myKid');
        $this->expectException(UnauthorizedException::class);
        $this->expectExceptionMessage("Invalid iss claim value: 'example.com'");
        $jwtAuthenticator->authenticate($jwt);
    }

    public function testInvalidAudClaim(): void
    {
        $jwtAuthenticator = new JwtAuthenticator([
            'myKid' => new Key(self::$publicKey, 'RS256'),
        ], ['example.com'], 'example.com');
        $jwt = JWT::encode([
            'aud' => 'fail.com',
            'iss' => 'example.com',
        ], self::$privateKey, 'RS256', 'myKid');
        $this->expectException(UnauthorizedException::class);
        $this->expectExceptionMessage("Invalid aud claim value: ");
        $jwtAuthenticator->authenticate($jwt);
    }
}
