<?php

declare(strict_types=1);

namespace Awardit\Microauth\Test;

use ArrayAccess;
use Awardit\Microauth\{
    Exception,
    JsonWebKeySet,
};
use Firebase\JWT\Key;
use GuzzleHttp\{
    Client,
    HandlerStack,
    Middleware,
};
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\Psr7\HttpFactory;
use Monolog\Logger;
use PHPUnit\Framework\TestCase;

/** @psalm-suppress UnusedClass */
class JsonWebKeySetTest extends TestCase
{
    public function testJsonWebKeySet(): void
    {
        $httpFactory = new HttpFactory();
        $responseContent = ['keys' => [[
            'kid' => 'myKid',
            'kty' => 'oct',
            'k'   => base64_encode('hello'),
            'alg' => 'RS256',
        ]]];
        $httpClient = $this->createMockClient(responses: [
            $httpFactory->createResponse(200)->withBody($httpFactory->createStream($this->json($responseContent))),
            $httpFactory->createResponse(200)->withBody($httpFactory->createStream($this->json($responseContent))),
        ]);
        $httpUri = $httpFactory->createUri('https://example.com/my-test-api');
        $logger = new Logger('microauth-test');

        $jsonWebKeySet = new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);
        $this->assertInstanceOf(ArrayAccess::class, $jsonWebKeySet);
        $this->assertTrue($jsonWebKeySet->hasCacheExpired());

        $keys = $jsonWebKeySet->getKeys();
        $this->assertCount(1, $keys);
        $this->assertInstanceOf(Key::class, $jsonWebKeySet['myKid']);
        $this->assertFalse($jsonWebKeySet->hasCacheExpired());
    }

    public function testInvalidSchemeError(): void
    {
        $httpFactory = new HttpFactory();
        $httpClient = $this->createMockClient();
        $httpUri = $httpFactory->createUri('http://example.com/my-test-api');
        $logger = new Logger('microauth-test');
        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Invalid JSON Web Key Set URL scheme 'http', expected 'https'");
        new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);
    }

    public function testServerError(): void
    {
        $httpFactory = new HttpFactory();
        $httpClient = $this->createMockClient(responses: [
            $httpFactory->createResponse(500),
        ]);
        $httpUri = $httpFactory->createUri('https://example.com/my-test-api');
        $logger = new Logger('microauth-test');

        $jsonWebKeySet = new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);

        $this->expectException(Exception::class);
        $this->expectExceptionMessage(
            "HTTP Error fetching JSON Web Key Set: 500 Internal Server Error for URL '{$httpUri}'"
        );
        $jsonWebKeySet->getKeys();
    }

    public function testParseError(): void
    {
        $httpFactory = new HttpFactory();
        $httpClient = $this->createMockClient(responses: [
            $httpFactory->createResponse(200),
        ]);
        $httpUri = $httpFactory->createUri('https://example.com/my-test-api');
        $logger = new Logger('microauth-test');

        $jsonWebKeySet = new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Failed to parse JSON Web Key Set body: Syntax error");
        $jsonWebKeySet->getKeys();
    }

    public function testMissingKeysError(): void
    {
        $httpFactory = new HttpFactory();
        $responseContent = 'error';
        $httpClient = $this->createMockClient(responses: [
            $httpFactory->createResponse(200)->withBody($httpFactory->createStream($this->json($responseContent))),
        ]);
        $httpUri = $httpFactory->createUri('https://example.com/my-test-api');
        $logger = new Logger('microauth-test');

        $jsonWebKeySet = new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Invalid JSON Web Key Set: Missing 'keys' list");
        $jsonWebKeySet->getKeys();
    }

    public function testInvalidKeysError(): void
    {
        $httpFactory = new HttpFactory();
        $responseContent = ['keys' => 'error'];
        $httpClient = $this->createMockClient(responses: [
            $httpFactory->createResponse(200)->withBody($httpFactory->createStream($this->json($responseContent))),
        ]);
        $httpUri = $httpFactory->createUri('https://example.com/my-test-api');
        $logger = new Logger('microauth-test');

        $jsonWebKeySet = new JsonWebKeySet($httpUri, $httpClient, $httpFactory, $logger);

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Invalid JSON Web Key Set: 'keys' is not a list");
        $jsonWebKeySet->getKeys();
    }

    /**
     * @param array<int, mixed> $responses
     */
    private function createMockClient(array $history = [], array $responses = []): Client
    {
        $historyMiddleware = Middleware::history($history);
        $mock = new MockHandler($responses);
        $handlerStack = HandlerStack::create($mock);
        $handlerStack->push($historyMiddleware);
        return new Client(['handler' => $handlerStack]);
    }

    private function json(mixed $in): string
    {
        return (string)json_encode($in);
    }
}
