<?php

declare(strict_types=1);

namespace Awardit\Microauth;

use ArrayAccess;
use DomainException;
use InvalidArgumentException;
use OutOfBoundsException;
use UnexpectedValueException;
use stdClass;
use Firebase\JWT\{
    BeforeValidException,
    ExpiredException,
    JWT,
    Key,
    SignatureInvalidException,
};

/**
 * JSON Web Token authenticator implementation.
 */
class JwtAuthenticator implements AuthenticatorInterface
{
    /**
     * String identifying an anonymous subject in an access token.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     */
    public const string ANONYMOUS_SUBJECT = "*";

    /**
     * @api
     */
    public function __construct(
        /**
         * Map of Key-id to Key implementation.
         *
         * Recommended to use an Awardit\Microauth\JsonWebKeySet with a URL
         * from configuration as code.
         *
         * If inter-process caching is required and a more standardized PHP
         * runtime is used where the process is restarted for each request,
         * look at Firebase\JWT\CachedKeySet.
         *
         * @var Key|ArrayAccess<string,Key>|array<string,Key> $keys
         */
        private $keys,
        /**
         * List of allowed access-key issuers, must match EXACTLY to the issuer claim.
         *
         * @var list<string>
         */
        private array $knownIssuers,
        /**
         * The identifier for the service, preferably the base-URL, will be
         * validated against the JWT audiences.
         */
        private string $audience,
    ) {
    }

    public function authenticate(string $token): TokenInterface
    {
        try {
            $decoded = JWT::decode($token, $this->keys);
        } catch (InvalidArgumentException $e) {
            // provided key/key-array is empty or malformed.
            throw new Exception("Invalid Authenticator key-configuration: " . $e->getMessage(), $e);
        } catch (DomainException $e) {
            // provided algorithm is unsupported OR
            // provided key is invalid OR
            // unknown error thrown in openSSL or libsodium OR
            // libsodium is required but not available.
            throw new Exception("Invalid Authenticator configuration: " . $e->getMessage(), $e);
        } catch (SignatureInvalidException $e) {
            // provided JWT signature verification failed.
            throw new UnauthorizedException("Signature verification failed", $e);
        } catch (BeforeValidException $e) {
            // provided JWT is trying to be used before "nbf" claim OR
            // provided JWT is trying to be used before "iat" claim.
            throw new ValidationException("Attempted to use token before nbf/iat claim", $e);
        } catch (ExpiredException $e) {
            // provided JWT is trying to be used after "exp" claim.
            throw new ValidationException("Attempted to use expired token", $e);
        } catch (OutOfBoundsException $e) {
            // CachedKeySet did not have the given kid
            throw new UnauthorizedException("Invalid JWT: " . $e->getMessage(), $e);
        } catch (UnexpectedValueException $e) {
            // provided JWT is malformed OR
            // provided JWT is missing an algorithm / using an unsupported algorithm OR
            // provided JWT algorithm does not match provided key OR
            // provided key ID in key/key-array is empty or invalid.
            throw new ValidationException("Invalid JWT: " . $e->getMessage(), $e);
        }

        return $this->createAccessToken($decoded);
    }

    /**
     * Creates an access token from a JSON Web Token payload. Will validate
     * issuer and audience.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     * @throws ValidationException if the token has an invalid shape
     * @throws ValidationException if the token does not have a valid issuer
     * @throws ValidationException if the token does not have a valid audience
     */
    protected function createAccessToken(stdClass $payload): AccessToken
    {
        // We are fine to do non-constant comparisons/validations here since the
        // payload has already had its signature validated.

        $iss = $this->getPayloadString($payload, "iss");
        $aud = $this->getPayloadList($payload, "aud");

        if (!in_array($iss, $this->knownIssuers, true)) {
            throw new UnauthorizedException(sprintf(
                "Invalid iss claim value: '%s'",
                $iss,
            ), context: [
                "token.iss" => $iss,
                "knownIssuers" => $this->knownIssuers,
            ]);
        }

        // TODO: More flexible audience-validation, allowing wildcards/supersets
        if (!in_array($this->audience, $aud, true)) {
            throw new UnauthorizedException(sprintf(
                "Invalid aud claim value: '%s'",
                (string)json_encode($aud),
            ), context: [
                "token.aud" => $aud,
                "audience" => $this->audience,
            ]);
        }

        $sub = $this->getPayloadString($payload, "sub");

        return new AccessToken(
            jti: $this->getPayloadString($payload, "jti"),
            sid: $this->getPayloadString($payload, "sid"),
            sub: $sub === self::ANONYMOUS_SUBJECT ? null : $sub,
            clientId: $this->getPayloadString($payload, "client_id"),
            iss: $iss,
            aud: $aud,
            iat: $this->getPayloadInt($payload, "iat"),
            exp: $this->getPayloadInt($payload, "exp"),
        );
    }

    /**
     * Fetches the value of a given key.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     * @throws ValidationException if the key does not exist
     */
    protected function getPayloadValue(stdClass $payload, string $key): mixed
    {
        if (!property_exists($payload, $key)) {
            throw new ValidationException(sprintf("Missing required key '%s'", $key));
        }

        return $payload->{$key};
    }

    /**
     * Fetches the value of a given key, ensuring it is an int.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     * @throws ValidationException if the key does not exist
     * @throws ValidationException if the value is not an int
     */
    protected function getPayloadInt(stdClass $payload, string $key): int
    {
        $value = $this->getPayloadValue($payload, $key);

        if (!is_int($value)) {
            throw new ValidationException(sprintf(
                "Expected key '%s' to contain int, got %s",
                $key,
                get_debug_type($value)
            ));
        }

        return $value;
    }

    /**
     * Fetches the value of a given key, ensuring it is a string.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     * @throws ValidationException if the key does not exist
     * @throws ValidationException if the value is not a string
     */
    protected function getPayloadString(stdClass $payload, string $key): string
    {
        $value = $this->getPayloadValue($payload, $key);

        if (!is_string($value)) {
            throw new ValidationException(sprintf(
                "Expected key '%s' to contain string, got %s",
                $key,
                get_debug_type($value)
            ));
        }

        return $value;
    }

    /**
     * Fetches the value of a given key, ensuring it contains a list of strings.
     *
     * Will convert a list of strings if the key contains a single string.
     *
     * @internal
     * @psalm-internal Awardit\Microauth
     * @throws ValidationException if the key does not exist
     * @throws ValidationException if the value is not a list of strings or a single string
     * @return list<string>
     */
    protected function getPayloadList(stdClass $payload, string $key): array
    {
        $value = $this->getPayloadValue($payload, $key);

        // JWT-claims with a list of strings allow a plain string as well in place of a list
        if (is_string($value)) {
            return [$value];
        }

        if (!is_array($value)) {
            throw new ValidationException(sprintf(
                "Expected key '%s' to contain list of strings, got %s",
                $key,
                get_debug_type($value)
            ));
        }

        if (!array_is_list($value)) {
            throw new ValidationException(sprintf(
                "Expected key '%s' to contain list of strings, got %s",
                $key,
                get_debug_type($value)
            ));
        }

        $list = [];

        foreach ($value as $v) {
            if (!is_string($v)) {
                throw new ValidationException(sprintf(
                    "Expected key '%s' to contain list of strings, got %s in list",
                    $key,
                    get_debug_type($v)
                ));
            }

            $list[] = $v;
        }

        return $list;
    }
}
