<?php

declare(strict_types=1);

namespace Awardit\Microauth;

use ArrayAccess;
use LogicException;
use OutOfBoundsException;
use Throwable;
use UnexpectedValueException;
use Firebase\JWT\{
    JWK,
    Key,
};
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\{
    RequestFactoryInterface,
    UriInterface,
};
use Psr\Log\LoggerInterface;

/**
 * JSON Web Key Set adapter with in-memory cacheing.
 *
 * Uses an off-thread fetching of the data if possible.
 *
 * The default Firebase CachedKeySet does not support refreshing or expiring
 * keys once they have been loaded into memory, they will only be loaded once
 * and will expire in the cache instead of in-memory.
 *
 * Another difference is that this key-set will never attempt to re-fetch the
 * key-set if the set is loaded but the supplied Key ID is not present. This way
 * we eliminate any risk of DDoS-attacks through this mechanism, otherwise
 * attackers could facilitate DoS-attacks by crafting JSON Web Tokens with
 * random values in the `kid` claim.
 *
 * @implements ArrayAccess<string, Key>
 * @api
 */
class JsonWebKeySet implements ArrayAccess
{
    /**
     * Fetched set of keys.
     *
     * @var Array<string, Key>
     */
    private ?array $keys = null;
    private ?int $lastFetchCompleted = null;

    public function __construct(
        /**
         * The full URL to the JSON Web Key Set, MUST be HTTPS.
         */
        private UriInterface $jwksUrl,
        /**
         * HTTP Client instance to use.
         */
        private ClientInterface $client,
        private RequestFactoryInterface $requestFactory,
        private LoggerInterface $logger,
        /**
         * Expiration time in seconds for a given key-set.
         *
         * Note that the key-set will never be re-fetched during this time,
         * even if a new `kid` value is observed in a token. Any unknown
         * `kid` values will be assumed to be invalid if not present in the
         * cached set.
         */
        private int $expiresAfter = 300,
    ) {
        if ($jwksUrl->getScheme() !== "https") {
            throw new Exception(sprintf(
                "Invalid JSON Web Key Set URL scheme '%s', expected 'https'",
                $jwksUrl->getScheme()
            ));
        }
    }

    /**
     * @param string $keyId
     */
    public function offsetGet($keyId): Key
    {
        $key = $this->getKeys()[$keyId] ?? null;

        if (!$key) {
            throw new OutOfBoundsException(sprintf("Key ID '%s' not found", $keyId));
        }

        return $key;
    }

    /**
     * @param string $keyId
     */
    public function offsetExists($keyId): bool
    {
        return array_key_exists($keyId, $this->getKeys());
    }

    /**
     * @param string $keyId
     * @param Key $value
     */
    public function offsetSet($keyId, $value): void
    {
        throw new LogicException("Method not implemented");
    }

    /**
     * @param string $keyId
     */
    public function offsetUnset($keyId): void
    {
        throw new LogicException("Method not implemented");
    }

    /**
     * @return Array<string, Key>
     */
    public function getKeys(): array
    {
        // TODO: Use a future here for swoole/async-PHP to eliminate
        // cache-storming and allow the cache to refresh off-thread

        if ($this->keys === null || $this->hasCacheExpired()) {
            $this->logger->debug("Fetching JSON Web Key Set");

            $this->keys = $this->refreshJWKS();
            $this->lastFetchCompleted = time();
        }

        return $this->keys;
    }

    public function hasCacheExpired(): bool
    {
        return $this->lastFetchCompleted === null ||
            $this->lastFetchCompleted + $this->expiresAfter < time();
    }

    /**
     * @internal
     * @psalm-internal Awardit\Microauth
     * @return Array<string, Key>
     */
    protected function refreshJWKS(): array
    {
        $request = $this->requestFactory->createRequest('GET', $this->jwksUrl);
        $response = $this->client->sendRequest($request);

        if ($response->getStatusCode() !== 200) {
            // This is not really unauthorized, we should throw 500 in this case
            $this->logger->error(
                "HTTP Error fetching JSON Web Key Set: {status} {reason} for URL '{url}'",
                [
                    "code" => $response->getStatusCode(),
                    "reason" => $response->getReasonPhrase(),
                    "url" => $this->jwksUrl,
                    "body" => (string)$response->getBody(),
                ]
            );

            throw new Exception(sprintf(
                "HTTP Error fetching JSON Web Key Set: %d %s for URL '%s'",
                $response->getStatusCode(),
                $response->getReasonPhrase(),
                $this->jwksUrl,
            ), context: [
                "code" => $response->getStatusCode(),
                "reason" => $response->getReasonPhrase(),
                "jwksUrl" => $this->jwksUrl,
                "body" => (string)$response->getBody(),
            ]);
        }

        $body = (string)$response->getBody();

        try {
            $data = json_decode($body, true, flags: JSON_THROW_ON_ERROR);
        } catch (Throwable $t) {
            $this->logger->error(
                "Failed to parse JSON Web Key Set body: {message}",
                [
                    "message" => $t->getMessage(),
                    "jwksUrl" => $this->jwksUrl,
                    "body" => $body,
                    "exception" => $t,
                ]
            );

            throw new Exception(
                "Failed to parse JSON Web Key Set body: " . $t->getMessage(),
                $t,
                [
                    "jwksUrl" => $this->jwksUrl,
                    "body" => $body,
                ]
            );
        }

        // Reimplementation of logic found in JWK::parseKeySet to make sure
        // the 'kid' claim is always set, and that no default algorithm is
        // used.
        if (!is_array($data) || !array_key_exists("keys", $data)) {
            throw new Exception("Invalid JSON Web Key Set: Missing 'keys' list");
        }

        if (!is_array($data["keys"]) || !array_is_list($data["keys"])) {
            throw new Exception("Invalid JSON Web Key Set: 'keys' is not a list");
        }

        $keys = [];

        foreach ($data["keys"] as $keyData) {
            // Discard unsupported keys and structures we do not understand for
            // resilience and forward-compatibility, but log them to make it
            // easier to debug
            if (!is_array($keyData)) {
                $this->logger->error("Invalid JSON Web Key Set item, not an object, skipping");

                continue;
            }

            if (!array_key_exists("kid", $keyData) || !is_string($keyData["kid"])) {
                $this->logger->error("Invalid JSON Web Key Set item, missing a valid 'kid' claim, skipping");

                continue;
            }

            $kid = $keyData["kid"];
            $key = JWK::parseKey($keyData, defaultAlg: null);

            if (!$key) {
                $this->logger->error("Unsupported key algorithm '{alg}' for kid '{kid}'", [
                    "alg" => (string)($keyData["alg"] ?? "null"),
                    "kid" => $kid,
                ]);

                continue;
            }

            $keys[$kid] = $key;
        }

        if (empty($keys)) {
            // Warn with error that we have no or no valid keys
            $this->logger->error("JSON Web Key Set contains none, or no valid, keys");
        }

        return $keys;
    }
}
