<?php

declare(strict_types=1);

namespace Awardit\Microauth\Test;

use Awardit\Microauth\JwtAuthenticator;
use Awardit\Microauth\Http\TokenAuthenticatorMiddleware;
use Awardit\Microauth\Test\Mock\RequestHandler;
use Firebase\JWT\{
    JWT,
    Key,
};
use GuzzleHttp\Psr7\HttpFactory;
use PHPUnit\Framework\TestCase;
use Psr\Http\Server\{
    MiddlewareInterface,
    RequestHandlerInterface,
};
use Psr\Log\NullLogger;

/** @psalm-suppress UnusedClass */
class TokenAuthenticatorTest extends TestCase
{
    private static string $privateKey;
    private static string $publicKey;
    private static HttpFactory $httpFactory;
    private static JwtAuthenticator $jwtAuthenticator;

    public static function setUpBeforeClass(): void
    {
        self::$privateKey = (string)file_get_contents(__DIR__ . '/Fixtures/privateKey.txt');
        self::$publicKey = (string)file_get_contents(__DIR__ . '/Fixtures/publicKey.txt');
        self::$httpFactory = new HttpFactory();
        self::$jwtAuthenticator = new JwtAuthenticator([
            'myKid' => new Key(self::$publicKey, 'RS256'),
        ], ['example.com'], 'example.com');
    }

    public function testTokenAuthenticator(): void
    {
        $middleware = new TokenAuthenticatorMiddleware(
            self::$jwtAuthenticator,
            self::$httpFactory,
            self::$httpFactory,
            new NullLogger(),
        );
        $jwt = JWT::encode([
            'aud' => 'example.com',
            'iss' => 'example.com',
            'sub' => 'user@example.com',
            'jti' => 'jti',
            'sid' => 'sid',
            'client_id' => 'client_id',
            'iat' => time(),
            'exp' => time() + 1000,
        ], self::$privateKey, 'RS256', 'myKid');
        $this->assertInstanceOf(MiddlewareInterface::class, $middleware);

        $request = self::$httpFactory
            ->createServerRequest("GET", "http://example.com/my-test-api")
            ->withHeader('Authorization', "Bearer {$jwt}");
        $handler = new RequestHandler();

        $response = $middleware->process($request, $handler);
        $this->assertEquals(200, $response->getStatusCode());
        $this->assertEquals('OK', $response->getReasonPhrase());
        $this->assertEmpty((string)$response->getBody());
    }

    public function testNoAuthorizationHeader(): void
    {
        $middleware = new TokenAuthenticatorMiddleware(
            self::$jwtAuthenticator,
            self::$httpFactory,
            self::$httpFactory,
            new NullLogger(),
        );
        $request = self::$httpFactory->createServerRequest("GET", "http://example.com/my-test-api");
        $handler = new RequestHandler();
        $response = $middleware->process($request, $handler);
        $this->assertEquals(401, $response->getStatusCode());
        $this->assertEquals('Unauthorized', $response->getReasonPhrase());
        $this->assertEquals('Unauthorized', (string)$response->getBody());
    }

    public function testBadAuthorizationHeader(): void
    {
        $middleware = new TokenAuthenticatorMiddleware(
            self::$jwtAuthenticator,
            self::$httpFactory,
            self::$httpFactory,
            new NullLogger(),
        );
        $request = self::$httpFactory
            ->createServerRequest("GET", "http://example.com/my-test-api")
            ->withHeader('Authorization', 'Invalid');
        $handler = new RequestHandler();
        $response = $middleware->process($request, $handler);
        $this->assertEquals(400, $response->getStatusCode());
        $this->assertEquals('Bad Request', $response->getReasonPhrase());
        $this->assertEquals('Bad Request', (string)$response->getBody());
    }

    public function testNotBearerAuthType(): void
    {
        $middleware = new TokenAuthenticatorMiddleware(
            self::$jwtAuthenticator,
            self::$httpFactory,
            self::$httpFactory,
            new NullLogger(),
        );
        $request = self::$httpFactory
            ->createServerRequest("GET", "http://example.com/my-test-api")
            ->withHeader('Authorization', 'Basic ABCD');
        $handler = new RequestHandler();
        $response = $middleware->process($request, $handler);
        $this->assertEquals(400, $response->getStatusCode());
        $this->assertEquals('Bad Request', $response->getReasonPhrase());
        $this->assertEquals('Invalid Authorization header value', (string)$response->getBody());
    }

    public function testInvalidBearerToken(): void
    {
        $middleware = new TokenAuthenticatorMiddleware(
            self::$jwtAuthenticator,
            self::$httpFactory,
            self::$httpFactory,
            new NullLogger(),
        );
        $request = self::$httpFactory
            ->createServerRequest("GET", "http://example.com/my-test-api")
            ->withHeader('Authorization', 'Bearer ABCD');
        $handler = new RequestHandler();
        $response = $middleware->process($request, $handler);
        $this->assertEquals(401, $response->getStatusCode());
        $this->assertEquals('Unauthorized', $response->getReasonPhrase());
        $this->assertEquals('Unauthorized', (string)$response->getBody());
    }
}
