<?php

declare(strict_types=1);

namespace Awardit\Microauth\Http;

use RuntimeException;
use Awardit\Microauth\{
    AuthenticatorInterface,
    TokenInterface,
    UnauthorizedException,
};
use Psr\Http\Message\{
    ResponseFactoryInterface,
    ResponseInterface,
    ServerRequestInterface,
    StreamFactoryInterface,
};
use Psr\Http\Server\{
    MiddlewareInterface,
    RequestHandlerInterface,
};
use Psr\Log\LoggerInterface;

/**
 * Middleware which requires authentication using a JSON Web Token.
 *
 * Upon successful authentication it will register the TokenInterface::class
 * attribute containing the validated token.
 *
 * @api
 */
class TokenAuthenticatorMiddleware implements MiddlewareInterface
{
    public function __construct(
        private AuthenticatorInterface $authenticator,
        private ResponseFactoryInterface $responseFactory,
        private StreamFactoryInterface $streamFactory,
        private LoggerInterface $logger
    ) {
    }

    public function process(
        ServerRequestInterface $request,
        RequestHandlerInterface $handler
    ): ResponseInterface {
        try {
            ["type" => $type, "token" => $token] = $this->getAuthorizationHeader($request);

            if ($type !== "bearer") {
                return $this->responseFactory
                    ->createResponse(400)
                    ->withHeader("Content-Type", "text/plain")
                    ->withBody($this->streamFactory->createStream("Invalid Authorization header value"));
            }

            $tokenInstance = $this->authenticator->authenticate($token);

            return $handler->handle($request->withAttribute(TokenInterface::class, $tokenInstance));
        } catch (BadRequestException $e) {
            $this->logger->debug($e->getMessage(), [
                ...$e->getContext(),
                "exception" => $e,
            ]);

            return $this->responseFactory
                ->createResponse(400)
                ->withHeader("Content-Type", "text/plain")
                ->withBody($this->streamFactory->createStream("Bad Request"));
        } catch (UnauthorizedException $e) {
            $this->logger->notice($e->getMessage(), [
                ...$e->getContext(),
                "exception" => $e,
            ]);
            return $this->responseFactory
                ->createResponse(401)
                ->withHeader("Content-Type", "text/plain")
                ->withBody($this->streamFactory->createStream("Unauthorized"));
        }
    }

    /**
     * @internal
     * @psalm-internal Awardit\Microauth
     * @return array{type:non-empty-lowercase-string, token:non-empty-string}
     */
    protected function getAuthorizationHeader(ServerRequestInterface $request): array
    {
        $values = $request->getHeader("Authorization");
        $numValues = count($values);

        if ($numValues === 0) {
            throw new UnauthorizedException("Missing required Authorization header");
        }

        if ($numValues > 1) {
            throw new BadRequestException("Too many Authorization header values");
        }

        $parts = explode(" ", $values[0], 2);

        if (count($parts) !== 2) {
            throw new BadRequestException("Invalid Authorization header value");
        }

        $type = strtolower(trim($parts[0]));
        $token = trim($parts[1]);

        if (empty($type) || empty($token)) {
            throw new BadRequestException("Invalid Authorization header value");
        }

        return [
            "type" => $type,
            "token" => $token,
        ];
    }
}
