<?php

declare(strict_types=1);

namespace Awardit\Microauth\Http;

use RuntimeException;
use Awardit\Microauth\AuthenticatorInterface;
use Awardit\Microauth\AuthorizationInterface;
use Awardit\Microauth\TokenInterface;
use Awardit\Microauth\UnauthorizedException;
use Nyholm\Psr7\Response;
use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\MiddlewareInterface;
use Psr\Http\Server\RequestHandlerInterface;
use Psr\Log\LoggerInterface;

/**
 * Middleware which requires authentication using a JSON Web Token.
 *
 * Upon successful authentication it will register the TokenInterface::class
 * attribute containing the validated token.
 *
 * @api
 */
class TokenAuthenticatorMiddleware implements MiddlewareInterface
{
    public function __construct(
        private AuthenticatorInterface $authenticator,
        private LoggerInterface $log
    ) {
    }

    public function process(
        ServerRequestInterface $request,
        RequestHandlerInterface $handler
    ): ResponseInterface {
        try {
            ["type" => $type, "token" => $token] = self::getAuthorizationHeader($request);

            if ($type !== "bearer") {
                return new Response(400, ["Content-Type" => "text/plain"], "Invalid Authorization header value");
            }

            $tokenInstance = $this->authenticator->authenticate($token);

            return $handler->handle($request->withAttribute(TokenInterface::class, $tokenInstance));
        } catch (BadRequestException $e) {
            $this->log->debug($e->getMessage(), [
                ...$e->getContext(),
                "exception" => $e,
            ]);

            return new Response(400, ["Content-Type" => "text/plain"], "Bad Request");
        } catch (UnauthorizedException $e) {
            $this->log->notice($e->getMessage(), [
                ...$e->getContext(),
                "exception" => $e,
            ]);

            return new Response(401, ["Content-Type" => "text/plain"], "Unauthorized");
        }
    }

    /**
     * @internal
     * @psalm-internal Awardit\Microauth
     * @return array{type:non-empty-lowercase-string, token:non-empty-string}
     */
    public static function getAuthorizationHeader(MessageInterface $message): array
    {
        $values = $message->getHeader("Authorization");
        $numValues = count($values);

        if ($numValues === 0) {
            throw new UnauthorizedException("Missing required Authorization header");
        }

        if ($numValues > 1) {
            throw new BadRequestException("Too many Authorization header values");
        }

        $parts = explode(" ", $values[0], 2);

        if (count($parts) !== 2) {
            throw new BadRequestException("Invalid Authorization header value");
        }

        $type = strtolower(trim($parts[0]));
        $token = trim($parts[1]);

        if (empty($type) || empty($token)) {
            throw new BadRequestException("Invalid Authorization header value");
        }

        return [
            "type" => $type,
            "token" => $token,
        ];
    }
}
