<?php

declare(strict_types=1);

namespace Awardit\MagentoPsr\Psr3;

use Mage;
use Psr\Log\{
    LoggerInterface,
    LoggerTrait,
    LogLevel,
};
use Zend_Log;
use Throwable;

use function DDTrace\current_context;

/**
 * PSR-3 Logger wrapper for Mage logging funciton.
 */
class Logger implements LoggerInterface
{
    use LoggerTrait;

    /**
     * Level ids to match magento.
     */
    public const LEVELS = [
        LogLevel::EMERGENCY => Zend_Log::EMERG,
        LogLevel::ALERT     => Zend_Log::ALERT,
        LogLevel::CRITICAL  => Zend_Log::CRIT,
        LogLevel::ERROR     => Zend_Log::ERR,
        LogLevel::WARNING   => Zend_Log::WARN,
        LogLevel::NOTICE    => Zend_Log::NOTICE,
        LogLevel::INFO      => Zend_Log::INFO,
        LogLevel::DEBUG     => Zend_Log::DEBUG,
    ];

    /**
     * Level names to match Magento.
     */
    public const LEVEL_NAMES = [
        LogLevel::EMERGENCY => "EMERG",
        LogLevel::ALERT     => "ALERT",
        LogLevel::CRITICAL  => "CRIT",
        LogLevel::ERROR     => "ERR",
        LogLevel::WARNING   => "WARN",
        LogLevel::NOTICE    => "NOTICE",
        LogLevel::INFO      => "INFO",
        LogLevel::DEBUG     => "DEBUG",
    ];

    /**
     * @var string
     */
    private $channel;

    /**
     * @var bool
     */
    private $debug;

    // TODO: make it easier to manage the debug flag without having to modify
    // the instantiation of the object.
    // Maybe global static configuration over which channels should use debug?
    public function __construct(string $channel, bool $debug = false)
    {
        $this->channel = $channel;
        $this->debug = $debug;
    }

    public function log($level, $message, array $context = []): void
    {
        // Skip backtrace for performance and log size
        $trace = $this->debug ? $this->trace(debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS)) : null;
        $context = $this->context($context, $trace);
        $record = [
            'level'         => self::LEVELS[$level] ?? Zend_Log::EMERG,
            'level_name'    => self::LEVEL_NAMES[$level] ?? "UNKNOWN",
            'channel'       => $this->channel,
            'message'       => $this->interpolate($message, $context),
            'context'       => $context,
            'extra'         => [],
        ];
        /** @psalm-suppress UndefinedFunction */
        if (extension_loaded('ddtrace')) {
            $record['dd'] = current_context();
        }
        error_log(json_encode(
            $record,
            JSON_INVALID_UTF8_SUBSTITUTE | JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
        ));
    }

    private function trace(array $trace): array
    {
        $result = [];
        for ($i = 0; $i < count($trace); $i++) {
            $result[] = [
                'class'     => $trace[$i + 1]['class'] ?? null,
                'function'  => $trace[$i + 1]['function'] ?? null,
                'location'  => [
                    'file'     => $trace[$i]['file'] ?? null,
                    'line'     => $trace[$i]['line'] ?? null,
                ],
            ];
        }
        return $result;
    }

    private function context(array $context, ?array $trace): array
    {
        if (!array_key_exists('storeCode', $context)) {
            $context['storeCode'] = Mage::app()->getCurrentStoreCode();
        }

        if ($trace) {
            $context['trace'] = $trace;
        }

        foreach ($context as $k => $v) {
            if ($v instanceof Throwable) {
                $context[$k] = $this->normalizeException($v);
            }
        }

        return $context;
    }

    private function interpolate(string $message, array $context): string
    {
        $replace = [];
        foreach ($context as $key => $value) {
            $replace['{' . $key . '}'] = $value;
        }
        return strtr($message, $replace);
    }

    /**
     * Normalizes an exception to a JSON-serializable structure which includes
     * stack-trace and previous exception.
     *
     * @return array{
     *   class:string,
     *   message:string,
     *   code:int,
     *   file:string,
     *   trace:Array<string>,
     *   previous?:array,
     * }
     */
    private function normalizeException(Throwable $e): array
    {
        $data = [
            'class' => get_class($e),
            'message' => $e->getMessage(),
            'code' => (int)$e->getCode(),
            'file' => $e->getFile() . ':' . $e->getLine(),
            'trace' => [],
        ];
        /**
         * @var array{line:int, file?:string} $frame
         */
        foreach ($e->getTrace() as $frame) {
            if (isset($frame['file'])) {
                $data['trace'][] = $frame['file'] . ':' . $frame['line'];
            }
        }
        if ($previous = $e->getPrevious()) {
            $data['previous'] = self::normalizeException($previous);
        }
        return $data;
    }
}
