<?php

declare(strict_types=1);

namespace Awardit\MagentoPsr\Psr3;

use Mage;
use Psr\Log\{
    InvalidArgumentException,
    LoggerInterface,
    LoggerTrait,
    LogLevel,
};
use Zend_Log;

use function DDTrace\current_context;

/**
 * PSR-3 Logger wrapper for Mage logging funciton.
 */
class Logger implements LoggerInterface
{
    use LoggerTrait;

    public const LEVELS = [
        Zend_Log::EMERG     => LogLevel::EMERGENCY,
        Zend_Log::ALERT     => LogLevel::ALERT,
        Zend_Log::CRIT      => LogLevel::CRITICAL,
        Zend_Log::ERR       => LogLevel::ERROR,
        Zend_Log::WARN      => LogLevel::WARNING,
        Zend_Log::NOTICE    => LogLevel::NOTICE,
        Zend_Log::INFO      => LogLevel::INFO,
        Zend_Log::DEBUG     => LogLevel::DEBUG,
    ];

    private $channel;

    public function __construct(string $channel)
    {
        $this->channel = $channel;
    }

    public function log($level, $message, array $context = []): void
    {
        $trace = $this->trace(debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS));
        $context = $this->context($context, $trace);
        $record = [
            'level'         => $this->level($level),
            'level_name'    => $level,
            'channel'       => $this->channel,
            'message'       => $this->interpolate($message, $context),
            'context'       => $context,
            'extra'         => [],
        ];
        /** @psalm-suppress UndefinedFunction */
        if (extension_loaded('ddtrace')) {
            $record['dd'] = current_context();
        }
        error_log(json_encode(
            $record,
            JSON_INVALID_UTF8_SUBSTITUTE | JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
        ));
    }

    private function level(string $level): int
    {
        $levelId = array_search($level, self::LEVELS, true);
        if ($levelId === false) {
            throw new InvalidArgumentException("Invalid logger level '{$level}'");
        }
        return $levelId;
    }

    private function trace(array $trace): array
    {
        $result = [];
        for ($i = 0; $i < count($trace); $i++) {
            $result[] = [
                'class'     => $trace[$i + 1]['class'] ?? null,
                'function'  => $trace[$i + 1]['function'] ?? null,
                'location'  => [
                    'file'     => $trace[$i]['file'] ?? null,
                    'line'     => $trace[$i]['line'] ?? null,
                ],
            ];
        }
        return $result;
    }

    private function context(array $context, array $trace): array
    {
        return array_merge([
            'storeCode' => Mage::app()->getCurrentStoreCode(),
            'trace'     => $trace,
        ], $context);
    }

    private function interpolate(string $message, array $context): string
    {
        $replace = [];
        foreach ($context as $key => $value) {
            $replace['{' . $key . '}'] = $value;
        }
        return strtr($message, $replace);
    }
}
