<?php

declare(strict_types=1);

namespace Awardit\MagentoPsr\Psr3;

use Awardit\MagentoPsr\Psr3\Context\{
    Context,
    ResolverInterface,
    DefaultResolver,
    ThrowableResolver
};
use Mage;
use Psr\Log\{
    LoggerInterface,
    LoggerTrait,
    LogLevel,
};
use Zend_Log;

use function DDTrace\current_context;

/**
 * PSR-3 Logger wrapper for Mage logging funciton.
 */
class Logger implements LoggerInterface
{
    use LoggerTrait;

    /**
     * Level ids to match magento.
     */
    public const LEVELS = [
        LogLevel::EMERGENCY => Zend_Log::EMERG,
        LogLevel::ALERT     => Zend_Log::ALERT,
        LogLevel::CRITICAL  => Zend_Log::CRIT,
        LogLevel::ERROR     => Zend_Log::ERR,
        LogLevel::WARNING   => Zend_Log::WARN,
        LogLevel::NOTICE    => Zend_Log::NOTICE,
        LogLevel::INFO      => Zend_Log::INFO,
        LogLevel::DEBUG     => Zend_Log::DEBUG,
    ];

    /**
     * Level names to match Magento.
     */
    public const LEVEL_NAMES = [
        LogLevel::EMERGENCY => "EMERG",
        LogLevel::ALERT     => "ALERT",
        LogLevel::CRITICAL  => "CRIT",
        LogLevel::ERROR     => "ERR",
        LogLevel::WARNING   => "WARN",
        LogLevel::NOTICE    => "NOTICE",
        LogLevel::INFO      => "INFO",
        LogLevel::DEBUG     => "DEBUG",
    ];

    private string $channel;
    private array $resolvers;

    public function __construct(string $channel, array $resolvers = [])
    {
        $this->channel = $channel;
        $this->resolvers = array_merge($resolvers, [
            new ThrowableResolver(),
            new DefaultResolver(),
        ]);
    }

    public function log($level, $message, array $context = []): void
    {
        $trace = $this->trace(debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS));
        $context = $this->resolveContext(array_merge([
            'storeCode'     => Mage::app()->getCurrentStoreCode(),
            'trace'         => $trace,
        ], $context));
        $record = [
            'level'         => self::LEVELS[$level] ?? Zend_Log::EMERG,
            'level_name'    => self::LEVEL_NAMES[$level] ?? "UNKNOWN",
            'channel'       => $this->channel,
            'message'       => $this->interpolate($message, $context),
            'context'       => $this->context($context),
            'extra'         => [],
        ];

        /** @psalm-suppress UndefinedFunction */
        if (extension_loaded('ddtrace')) {
            $record['dd'] = current_context();
        }
        error_log(json_encode(
            $record,
            JSON_INVALID_UTF8_SUBSTITUTE | JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
        ));
    }

    private function trace(array $trace): array
    {
        $result = [];
        for ($i = 0; $i < count($trace); $i++) {
            $result[] = [
                'class'     => $trace[$i + 1]['class'] ?? null,
                'function'  => $trace[$i + 1]['function'] ?? null,
                'location'  => [
                    'file'     => $trace[$i]['file'] ?? null,
                    'line'     => $trace[$i]['line'] ?? null,
                ],
            ];
        }
        return $result;
    }

    private function resolveContext(array $context): array
    {
        return array_map(function ($item) {
            foreach ($this->resolvers as $resolver) {
                if ($resolver->canResolve($item)) {
                    return $resolver->resolve($item);
                }
            }
        }, $context);
    }

    private function context(array $context): array
    {
        return array_map(
            /** @return mixed */
            function (Context $item) {
                return $item->jsonSerialize();
            },
            $context
        );
    }

    private function interpolate(string $message, array $context): string
    {
        $replace = [];
        foreach ($context as $key => $value) {
            $replace['{' . $key . '}'] = (string)$value;
        }
        return strtr($message, $replace);
    }
}
