<?php

declare(strict_types=1);

namespace Awardit\MagentoPsr\Psr3\Context;

use Mage_Core_Exception;
use Throwable;

/**
 * Log resolver for Throwable.
 */
class ThrowableResolver implements ResolverInterface
{
    /** @param mixed $context */
    public function canResolve($context): bool
    {
        return $context instanceof Throwable;
    }

    /** @param mixed $context */
    public function resolve($context): Context
    {
        return new Context($context->getMessage(), $this->normalize($context));
    }

    /**
     * Normalizes an exception to a JSON-serializable structure which includes
     * stack-trace and previous exception.
     *
     * @return object{
     *   class:class-string<Throwable>,
     *   message:string,
     *   code:int|string,
     *   file:non-falsy-string,
     *   trace:array<empty, empty>,
     * }
     */
    private function normalize(Throwable $e): object
    {
        $data = (object)[
            'class' => get_class($e),
            'message' => $e->getMessage(),
            'code' => $e->getCode(),
            'file' => $e->getFile() . ':' . $e->getLine(),
            'trace' => [],
        ];
        /**
         * @var array{line:int, file?:string} $frame
         */
        foreach ($e->getTrace() as $frame) {
            if (isset($frame['file'])) {
                $data->trace[] = $frame['file'] . ':' . $frame['line'];
            }
        }
        if ($previous = $e->getPrevious()) {
            $data->previous = $this->normalize($previous);
        }
        if ($e instanceof Mage_Core_Exception) {
            foreach ($e->getMessages() as $message) {
                $data->messages[] = (object)[
                    'type' => $message->getType(),
                    'message' => $message->getText(),
                ];
            }
        }
        return $data;
    }
}
