<?php

declare(strict_types=1);

/**
 * @psalm-type Product array{id:string, sku:string, type:string, name:string, description:string, image:string, price:string, status:string, tax_class_id:string, categories:string, nestedNonVirtual:int, gallery:array}
 * @psalm-type ProductSelection array{id:string, defaultQty:string}
 */
class Awardit_OrderApi_Model_Product {
    const SEPARATOR = "±";

    /**
     * @param array{limit:int, offset:int, enabled?:boolean} $options
     * @return list<Product>
     */
    public function getProducts(Mage_Core_Model_Store $store, array $options): array {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $select = $this->select($store, $connection);

        $select->order("p.entity_id asc");

        if(array_key_exists("enabled", $options)) {
            $enabledValue = $options["enabled"] ? Mage_Catalog_Model_Product_Status::STATUS_ENABLED : Mage_Catalog_Model_Product_Status::STATUS_DISABLED;

            $select->where("status.value = ? OR (status.value IS NULL AND status_g.value = ?)", $enabledValue);
        }

        $select->limit(
            max(min($options["limit"], 500), 1),
            max($options["offset"], 0)
        );

        /** @var list<Product> */
        $products = array_values($connection->fetchAll($select));

        // Get gallery images for each product
        foreach ($products as &$product) {
            $product['gallery'] = $this->getGalleryImages($product['id']);
        }

        return $products;
    }

    /**
     * @return ?Product
     */
    public function getProduct(Mage_Core_Model_Store $store, string $sku): ?array {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $select = $this->select($store, $connection);

        $select->where("p.sku = ?", $sku);
        $select->limit(1);

        /** @var ?Product */
        $product = array_values($connection->fetchAll($select))[0] ?? null;

        if ($product) {
            $product['gallery'] = $this->getGalleryImages($product['id']);
        }

        return $product;
    }

    /**
     * Get gallery images for a product
     * @param string|int $productId
     * @return array<array{label:string|null, url:string}>
     */
    private function getGalleryImages($productId): array {
        // Get all images using Magento's product model
        $product = Mage::getModel('catalog/product')->load($productId);
        $mediaGallery = $product->getMediaGalleryImages();

        $uniqueImages = [];

        // Add all images including the main image
        foreach ($mediaGallery as $image) {
            $uniqueImages[$image->getFile()] = [
                'label' => (string)$image->getLabel() ?: null,
                'url' => (string)$image->getFile()
            ];
        }

        return array_values($uniqueImages);
    }

    private function select(
        Mage_Core_Model_Store $store,
        Varien_Db_Adapter_Interface $connection
    ): Zend_Db_Select {
        $select = $connection->select();
        $resource = Mage::getSingleton("core/resource");
        $websiteId = (int)$store->getWebsiteId();

        $select->from([
            "p" => $resource->getTableName("catalog/product"),
        ], [
            "id" => "entity_id",
            "sku" => "sku",
            "type" => "type_id",
            "required_options" => "required_options",
        ]);

        // Only show products assigned to the website
        $select->join([
            "w" => $resource->getTableName("catalog/product_website")
        ], "w.product_id = p.entity_id AND w.website_id = {$websiteId}", []);

        $this->joinAttribute($store, $select, "name");
        $this->joinAttribute($store, $select, "description");
        $this->joinAttribute($store, $select, "image");
        $this->joinAttribute($store, $select, "price");
        $this->joinAttribute($store, $select, "status");
        $this->joinAttribute($store, $select, "tax_class_id");

        $categories = $this->selectCategoryNames($store, $connection);

        $categories->join([
            "cp" => $resource->getTableName("catalog/category_product"),
        ], "c.entity_id = cp.category_id", []);
        $categories->where("cp.product_id = p.entity_id");

        // We need to fetch the nested product types to see if bundles are virtual or not
        $nestedNonVirtual = $connection->select();
        $nestedNonVirtual->from([
            "chp" => $resource->getTableName("catalog/product"),
        ], []);
        $nestedNonVirtual->join([
            "pr" => $resource->getTableName("catalog/product_relation"),
        ], "pr.child_id = chp.entity_id", []);
        $nestedNonVirtual->columns([
            "product_types" => new Zend_Db_Expr("COUNT(1)"),
        ]);
        $nestedNonVirtual->where("pr.parent_id = p.entity_id AND chp.type_id <> 'virtual'");

        // Count the bundle options which do NOT have any default options
        $requiredOptions = $connection->select();

        $requiredOptions->from([
            "bo" => $resource->getTableName("bundle/option"),
        ], ["option_id"]);
        $requiredOptions->joinLeft([
            "bs" => $resource->getTableName("bundle/selection"),
        ], "bs.option_id = bo.option_id AND bs.is_default = 1 AND bs.selection_qty > 0", ["selection_id"]);
        $requiredOptions->where("bo.required = 1 AND bo.parent_id = p.entity_id");
        $requiredOptions->group("bo.option_id");
        $requiredOptions->having("COUNT(bs.is_default) = 0");

        $select->columns([
            "categories" => $categories,
            "nestedNonVirtual" => $nestedNonVirtual,
        ]);

        // We do not allow setting product options when ordering, filter out
        // any simple/virtual which rquires options, and then the same for bundle
        /** @psalm-suppress ImplicitToStringCast */
        $select->where(new Zend_Db_Expr("(p.type_id IN ('simple', 'virtual') AND p.required_options = 0 OR p.type_id = 'bundle' AND NOT EXISTS ($requiredOptions))"));

        return $select;
    }

    private function selectCategoryNames(
        Mage_Core_Model_Store $store,
        Varien_Db_Adapter_Interface $connection
    ): Zend_Db_Select {
        $config = Mage::getSingleton("eav/config");
        $resource = Mage::getSingleton("core/resource");
        $storeId = (int)$store->getId();
        $attr = $config->getAttribute("catalog_category", "name");

        assert($attr !== false);

        $attrId = (int)$attr->getId();
        /** @var string */
        $backend = $attr->getBackendType();
        $table = $resource->getTableName("catalog/category")."_".$backend;

        $select = $connection->select();

        $select->from([
            "c" => $resource->getTableName("catalog/category"),
        ], []);

        $select->joinLeft(["csv" => $table], "csv.store_id = {$storeId} AND csv.entity_id = c.entity_id AND csv.attribute_id = {$attrId}", []);
        $select->joinLeft(["cgv" => $table], "cgv.store_id = 0 AND cgv.entity_id = c.entity_id AND cgv.attribute_id = {$attrId}", []);

        $select->columns([
            "categories" => new Zend_Db_Expr("GROUP_CONCAT(COALESCE(csv.value, cgv.value) SEPARATOR '".self::SEPARATOR."')"),
        ]);

        // Make sure we only use the categories available in the store
        $select->where("c.path LIKE '1/".(int)$store->getRootCategoryId()."%'");

        return $select;
    }

    private function joinAttribute(
        Mage_Core_Model_Store $store,
        Zend_Db_Select $select,
        string $attributeCode
    ): void {
        $config = Mage::getSingleton("eav/config");
        $resource = Mage::getSingleton("core/resource");
        $storeId = (int)$store->getId();
        $attr = $config->getAttribute("catalog_product", $attributeCode);

        assert($attr !== false);

        $attrId = (int)$attr->getId();
        /** @var string */
        $backend = $attr->getBackendType();
        $table = $resource->getTableName("catalog/product")."_".$backend;

        $select->columns([
            $attributeCode => new Zend_Db_Expr("COALESCE({$attributeCode}.value, {$attributeCode}_g.value, '')"),
        ]);
        $select->joinLeft([$attributeCode => $table], "{$attributeCode}.store_id = {$storeId} AND {$attributeCode}.entity_id = p.entity_id AND {$attributeCode}.attribute_id = {$attrId}", []);
        $select->joinLeft([$attributeCode."_g" => $table], "{$attributeCode}_g.store_id = 0 AND {$attributeCode}_g.entity_id = p.entity_id AND {$attributeCode}_g.attribute_id = {$attrId}", []);
    }

    /**
     * @psalm-suppress PossiblyUnusedParam
     */
    public function getDefaultSelection(Mage_Core_Model_Store $store, int $entityId): array {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $defaultOptions = $connection->select();

        $defaultOptions->from([
            "bo" => $resource->getTableName("bundle/option"),
        ], []);
        $defaultOptions->join([
            "bs" => $resource->getTableName("bundle/selection"),
        ], "bs.option_id = bo.option_id AND bs.is_default = 1 AND bs.selection_qty > 0", [
            "productId" => "product_id",
            "defaultQty" => "selection_qty",
        ]);
        $defaultOptions->where("bs.is_default = 1 AND bo.parent_id = ?", $entityId);

        /** @var list<ProductSelection> */
        return array_values($connection->fetchAll($defaultOptions));
    }
}
