<?php

declare(strict_types=1);

/**
 * @psalm-import-type Product from Awardit_OrderApi_Model_Product
 */
class Awardit_OrderApi_ProductsController extends Awardit_OrderApi_Controller_Base {
    public function getAll(): array {
        $store = Mage::app()->getStore();
        $productResource = Mage::getModel("awardit_orderapi/product");
        $stockCollection = Mage::getResourceModel("cataloginventory/stock_item_collection");
        $taxCalculator = Mage::getSingleton("tax/calculation");
        $request = $this->getRequest();

        $options = [
            "limit" => (int)($request->getQuery("limit") ?: "100"),
            "offset" => (int)($request->getQuery("offset") ?: "0"),
        ];

        $enabled = strtolower(trim((string)$request->getQuery("enabled") ?: ""));

        if(in_array($enabled, ["true", "false"])) {
            $options["enabled"] = $enabled === "true";
        }

        $data = $productResource->getProducts($store, $options);

        $items = [];
        $ids = [];
        $bundles = [];

        foreach($data as $p) {
            $ids[] = (int)$p["id"];
            $items[(int)$p["id"]] = $this->convertItem($store, $taxCalculator, $p);

            if ($p["type"] === "bundle") {
                $defaultQtys = $productResource->getDefaultSelection($store, (int)$p["id"]);

                $bundles[(int)$p["id"]] = $defaultQtys;

                foreach($defaultQtys as $selection) {
                    $ids[] = (int)$selection["productId"];
                }
            }
        }

        $stockCollection->addProductsFilter(array_unique($ids));
        $stockCollection->joinStockStatus($store->getId());

        foreach($stockCollection as $stockItem) {
            $productId = (int)$stockItem->getProductId();

            if( ! array_key_exists($productId, $items)) {
                continue;
            }

            if( ! $stockItem->getManageStock()) {
                continue;
            }

            $items[$productId]["stock"] = [
                "inStock" => (bool)$stockItem->getIsInStock(),
                "qty" => (int)$stockItem->getQty(),
            ];
        }

        foreach($bundles as $productId => $selections) {
            $qty = PHP_INT_MAX;

            foreach($selections as $s) {
                foreach($stockCollection as $stockItem) {
                    if ($stockItem->getProductId() == $s["productId"]) {
                        $qty = min($qty, floor((float)$stockItem->getQty() / (float)$s["defaultQty"]));
                    }
                }
            }

            if($qty !== PHP_INT_MAX) {
                $items[$productId]["stock"]["qty"] = (int)max(0, $qty);
            }
        }

        return [200, array_values($items)];
    }

    public function getItem(string $id): array {
        $store = Mage::app()->getStore();
        $productResource = Mage::getModel("awardit_orderapi/product");
        $stockCollection = Mage::getResourceModel("cataloginventory/stock_item_collection");
        $taxCalculator = Mage::getSingleton("tax/calculation");

        $data = $productResource->getProduct($store, $id);

        if( ! $data) {
            return [404];
        }

        $stockCollection->addProductsFilter([$data["id"]]);
        $stockCollection->joinStockStatus($store->getId());

        $item = $this->convertItem($store, $taxCalculator, $data);

        foreach($stockCollection as $stockItem) {
            if( ! $stockItem->getManageStock()) {
                continue;
            }

            $item["stock"] = [
                "inStock" => (bool)$stockItem->getIsInStock(),
                "qty" => (int)$stockItem->getQty(),
            ];
        }

        return [200, $item];
    }

    private function getMediaUrl(Mage_Core_Model_Store $store, string $image): string {
        $mediaBaseUrl = (string)$store->getConfig("web/secure/base_media_url");

        return $mediaBaseUrl . implode("/", ["catalog", "product", trim($image, "/")]);
    }

    /**
     * @param Product $p
     */
    private function convertItem(
        Mage_Core_Model_Store $store,
        Mage_Tax_Model_Calculation $taxCalculator,
        array $p
    ): array {
        $countryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);
        $customerTaxClass = $taxCalculator->getDefaultCustomerTaxClass($store);

        // Convert gallery images to include full URLs
        $gallery = array_map(function($image) use ($store) {
            return [
                'label' => $image['label'],
                'url' => $this->getMediaUrl($store, $image['url'])
            ];
        }, $p['gallery'] ?? []);

        return [
            "sku" => $p["sku"],
            "name" => $p["name"],
            "description" => $p["description"],
            "image" => $p["image"] ? [ "url" => $this->getMediaUrl($store, $p["image"]) ] : null,
            "gallery" => $gallery,
            "price" => [
                "value" => (float)$p["price"],
                "vatPercent" => $taxCalculator->getRate(new Varien_Object([
                    "country_id" => $countryId,
                    "customer_class_id" => $customerTaxClass,
                    "product_class_id" => $p["tax_class_id"],
                ])),
            ],
            "virtual" => $this->isProductVirtual($p),
            "enabled" => $p["status"] == Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "categories" => $this->convertProductCategories($p["categories"]),
            "stock" => null,
        ];
    }

    /**
     * @param Product $p
     */
    private function isProductVirtual(array $p): bool {
        if ($p["type"] === "bundle") {
            return (int)$p["nestedNonVirtual"] === 0;
        }
        else if ($p["type"] !== "virtual") {
            return false;
        }

        return true;
    }

    /**
     * @return list<array{name:string}>
     */
    private function convertProductCategories(?string $categories): array {
        if(empty($categories)) {
            return [];
        }

        return array_values(array_filter(array_map(function(string $categoryName): ?array {
            $categoryName = trim($categoryName);

            if( ! $categoryName) {
                return null;
            }

            return [
                "name" => $categoryName,
            ];
        }, explode(Awardit_OrderApi_Model_Product::SEPARATOR, $categories))));
    }
}
