<?php

declare(strict_types=1);

namespace Awardit\OrderApi;

use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Sales_Model_Order;
use Mage_Sales_Model_Order_Address;
use Mage_Sales_Model_Order_Item;
use Points_Core_PointPaymentRequiredException;
use Throwable;
use Varien_Object;

use Points\Core\Extension\Quote;
use Spatie\Snapshots\MatchesSnapshots;
use Crossroads\Magento\Test\Integration\Request;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;

use PHPUnit\Framework\TestCase;

class ProductsControllerTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();

        Config::setConfigData("awardit_orderapi/general/enabled", "1");
        Config::setConfigData("awardit_orderapi/general/api_key", "thekey");
    }

    public function tearDown(): void {
        MagentoManager::reset();
        Config::unsetConfigData("awardit_orderapi/general/enabled");
        Config::unsetConfigData("awardit_orderapi/general/api_key");
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    public function testProducts(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductsLimit(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products?limit=3", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductsLimitOffset(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products?limit=3&offset=2", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductItemNotFound(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/asd123", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertEquals(404, $resp->getHttpResponseCode());
    }

    public function testProductVirtual(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-virtual", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductSimple(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-simple", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductBundleWithDefaults(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-bundle-default", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductBundleWithRequiredOptions(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-bundle-select", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertEquals(404, $resp->getHttpResponseCode());
    }

    public function testProductBundleWithHiddenOptions(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-bundle-hidden", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductCustomOptionsOptional(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-custom-options", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductCustomOptionsRequired(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-custom-options-required", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertEquals(404, $resp->getHttpResponseCode());
    }

    public function testProductBundleVirtual(): void {
        $resp = MagentoManager::runRequest(new Request("GET /awardit_orderapi/products/test-bundle-virtual", [
            "Authorization" => "Bearer thekey",
        ]));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
