<?php

declare(strict_types=1);

namespace Awardit\OrderApi;

use Exception;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Sales_Model_Order;
use Mage_Sales_Model_Order_Address;
use Mage_Sales_Model_Order_Item;
use Points_Core_PointPaymentRequiredException;
use Throwable;
use Varien_Object;

use Awardit\OrderApi\Extension\Order;
use Points\Core\Extension\Quote;
use Spatie\Snapshots\MatchesSnapshots;
use Crossroads\Magento\Test\Integration\Request;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;

use PHPUnit\Framework\TestCase;

class OrderTest extends TestCase {
    use MatchesSnapshots;

    const ORDER_DIFF_KEYS = [
        "applied_rule_ids" => true,
        "billing_address_id" => true,
        "created_at" => true,
        "customer" => true,
        "customer_group_id" => true,
        "customer_id" => true,
        "customer_tax_class_id" => true,
        "customer_taxvat" => true,
        "entity_id" => true,
        "increment_id" => true,
        "protect_code" => true,
        "quote" => true,
        "quote_id" => true,
        "shipping_address_id" => true,
        "store_id" => true,
        "updated_at" => true,
    ];

    const ORDER_ADDRESS_DIFF_KEYS = [
        "store_id" => true,
        "customer_id" => true,
        "customer_address_id" => true,
        "vat_id" => true,
        "vat_request_id" => true,
        "region_id" => true,
        "parent_id" => true,
        "created_at" => true,
        "updated_at" => true,
        "entity_id" => true,
    ];

    const ORDER_ITEM_DIFF_KEYS = [
        "created_at" => true,
        "item_id" => true,
        "order_id" => true,
        "product" => true,
        "product_id" => true,
        "product_options" => true,
        "quote_item_id" => true,
        "quote_parent_item_id" => true,
        "store_id" => true,
        "updated_at" => true,
    ];

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        Config::unsetConfigData("awardit_orderapi/general/enabled");
        Config::unsetConfigData("awardit_orderapi/general/api_key");

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->save();

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    protected function clearQuotes(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        try {
            $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

            $setupModel->deleteConfigData("carriers/freeshipping/active", "websites");
            $setupModel->deleteConfigData("carriers/freeshipping/free_shipping_subtotal", "websites");

            // Prevent any existing quotes from being loaded
            $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

            if($conn) {
                $conn->query("UPDATE sales_flat_quote SET is_active = 0");
            }
        }
        catch(Exception $e) {
            error_log((string)$e);
        }

        MagentoManager::reset();
    }

    protected function assertOrderMatchesSnapshot(Mage_Sales_Model_Order $order): void {
        $items = $order->getAllItems();

        /**
         * @var Array<string, mixed>
         */
        $data = $order->getData();

        $orderData = array_diff_key($data, self::ORDER_DIFF_KEYS);
        $addressData = array_map(function(Mage_Sales_Model_Order_Address $a): array {
            return array_diff_key($a->getData(), self::ORDER_ADDRESS_DIFF_KEYS);
        }, $order->getAddressesCollection()->getItems());
        $itemData = array_map(function(Mage_Sales_Model_Order_Item $i): array {
            return array_diff_key($i->getData(), self::ORDER_ITEM_DIFF_KEYS);
        }, $items);

        $this->assertMatchesYamlSnapshot([
            "order" => $orderData,
            "addresses" => $addressData,
            "items" => $itemData,
        ]);
    }

    public function testPlaceOrderVirtual(): void {
        Config::setConfigData("awardit_orderapi/general/enabled", "1");
        Config::setConfigData("awardit_orderapi/general/api_key", "thekey");

        $resp = MagentoManager::runRequest(new Request("POST /awardit_orderapi/orders", [
            "Content-Type" => "application/json",
            "Authorization" => "Bearer thekey",
        ], json_encode([
            "items" => [
                [
                    "sku" => "test-virtual",
                    "qty" => 1,
                ]
            ],
            "recipient" => [
                "email" => "test@example.com",
                "telephone" => "0123456789",
            ],
        ])));

        /** @var ?Order */
        $order = Mage::registry("current_order");

        $this->assertNotNull($order);

        $body = json_decode($resp->getBody(), true);

        $this->assertIsArray($body);
        $this->assertArrayHasKey("orderId", $body);
        $this->assertEquals($order->getIncrementId(), $body["orderId"]);
        $this->assertArrayHasKey("status", $body);
        $this->assertEquals("pending", $body["status"]);
        $this->assertArrayHasKey("parcels", $body);
        $this->assertEquals([], $body["parcels"]);

        $this->assertEquals(201, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertOrderMatchesSnapshot($order);
    }

    public function testPlaceOrderVirtual2(): void {
        Config::setConfigData("awardit_orderapi/general/enabled", "1");
        Config::setConfigData("awardit_orderapi/general/api_key", "thekey");

        $resp = MagentoManager::runRequest(new Request("POST /awardit_orderapi/orders", [
            "Content-Type" => "application/json",
            "Authorization" => "Bearer thekey",
        ], json_encode([
            "items" => [
                [
                    "sku" => "test-virtual",
                    "qty" => 2,
                ]
            ],
            "recipient" => [
                "email" => "test@example.com",
                "telephone" => "0123456789",
            ],
        ])));

        /** @var ?Order */
        $order = Mage::registry("current_order");

        $this->assertNotNull($order);

        $body = json_decode($resp->getBody(), true);

        $this->assertIsArray($body);
        $this->assertArrayHasKey("orderId", $body);
        $this->assertEquals($order->getIncrementId(), $body["orderId"]);
        $this->assertArrayHasKey("status", $body);
        $this->assertEquals("pending", $body["status"]);
        $this->assertArrayHasKey("parcels", $body);
        $this->assertEquals([], $body["parcels"]);

        $this->assertEquals(201, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $this->assertOrderMatchesSnapshot($order);
    }

    public function testPlaceOrderVirtualQtyNotAvailable(): void {
        MagentoManager::initAdmin();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 1,
                "qty" => 1,
            ])
            ->save();

        MagentoManager::reset();

        Config::setConfigData("awardit_orderapi/general/enabled", "1");
        Config::setConfigData("awardit_orderapi/general/api_key", "thekey");

        $resp = MagentoManager::runRequest(new Request("POST /awardit_orderapi/orders", [
            "Content-Type" => "application/json",
            "Authorization" => "Bearer thekey",
        ], json_encode([
            "items" => [
                [
                    "sku" => "test-virtual",
                    "qty" => 2,
                ]
            ],
            "recipient" => [
                "email" => "test@example.com",
                "telephone" => "0123456789",
            ],
        ])));

        $body = json_decode($resp->getBody(), true);

        $this->assertIsArray($body);
        $this->assertArrayHasKey("code", $body);
        $this->assertEquals("productQuantityNotAvailable", $body["code"]);
        $this->assertArrayHasKey("message", $body);
        $this->assertEquals("Product with SKU 'test-virtual' does not have the requested quantity available", $body["message"]);
        $this->assertArrayHasKey("data", $body);
        $this->assertEquals(["sku" => "test-virtual"], $body["data"]);

        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrderVirtualOutOfStock(): void {
        MagentoManager::initAdmin();

        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        Mage::getModel("cataloginventory/stock_item")
            ->loadByProduct($product)
            ->addData([
                "is_in_stock" => 0,
                "qty" => 999,
            ])
            ->save();

        MagentoManager::reset();

        Config::setConfigData("awardit_orderapi/general/enabled", "1");
        Config::setConfigData("awardit_orderapi/general/api_key", "thekey");

        $resp = MagentoManager::runRequest(new Request("POST /awardit_orderapi/orders", [
            "Content-Type" => "application/json",
            "Authorization" => "Bearer thekey",
        ], json_encode([
            "items" => [
                [
                    "sku" => "test-virtual",
                    "qty" => 2,
                ]
            ],
            "recipient" => [
                "email" => "test@example.com",
                "telephone" => "0123456789",
            ],
        ])));

        $body = json_decode($resp->getBody(), true);

        $this->assertIsArray($body);
        $this->assertArrayHasKey("code", $body);
        $this->assertEquals("productOutOfStock", $body["code"]);
        $this->assertArrayHasKey("message", $body);
        $this->assertEquals("Product with SKU 'test-virtual' is out of stock", $body["message"]);
        $this->assertArrayHasKey("data", $body);
        $this->assertEquals(["sku" => "test-virtual"], $body["data"]);

        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
