<?php

declare(strict_types=1);

class Awardit_OrderApi_Helper_Data extends Mage_Core_Helper_Abstract {
    const LOG_CHANNEL = "awardit_orderapi";

    public function isEnabled(Mage_Core_Model_Store $store): bool {
        return (bool)$store->getConfig("awardit_orderapi/general/enabled");
    }

    public function getApiToken(Mage_Core_Model_Store $store): string {
        return (string)$store->getConfig("awardit_orderapi/general/api_key");
    }

    public function getShippingMethod(Mage_Core_Model_Store $store): ?string {
        $val = $store->getConfig("awardit_orderapi/general/shipping");

        return $val ? (string)$val : null;
    }

    public function ensureAccess(
        Mage_Core_Model_Store $store,
        Mage_Core_Controller_Request_Http $request
    ): void {
        if( ! $this->isEnabled($store)) {
            Mage::log("The module is not enabled", Zend_Log::DEBUG, self::LOG_CHANNEL);

            throw new Awardit_OrderApi_DisabledException();
        }

        $authorizationData = $request->getHeader("Authorization");

        if( ! $authorizationData) {
            throw new Awardit_OrderApi_UnauthorizedException();
        }

        $parts = explode(" ", trim($authorizationData));

        if(count($parts) !== 2) {
            throw new Awardit_OrderApi_UnauthorizedException();
        }

        if(strtolower($parts[0]) !== "bearer") {
            throw new Awardit_OrderApi_UnauthorizedException();
        }

        if($parts[1] !== $this->getApiToken($store)) {
            throw new Awardit_OrderApi_UnauthorizedException();
        }
    }

    /**
     * Returns true if the supplied product is visible on its own in the
     * given store.
     *
     * NOTE: Actually ignores visibility on purpose. We only look on website
     * and enabled/disabled.
     */
    public function isProductVisible(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): bool {
        return $product->getId() &&
            ! $product->isDisabled() &&
            in_array($store->getWebsiteId(), $product->getWebsiteIds());
    }
}
