<?php

declare(strict_types=1);

/**
 * @psalm-import-type Product from Awardit_OrderApi_Model_Product
 */
class Awardit_OrderApi_ProductsController extends Awardit_OrderApi_Controller_Base {
    public function getAll(): array {
        $store = Mage::app()->getStore();
        $productResource = Mage::getModel("awardit_orderapi/product");
        $stockCollection = Mage::getResourceModel("cataloginventory/stock_item_collection");
        $taxCalculator = Mage::getSingleton("tax/calculation");

        $data = $productResource->getProducts($store, [
            "limit" => $this->getRequest()->getQuery("limit"),
            "offset" => $this->getRequest()->getQuery("offset"),
        ]);

        $items = [];
        $ids = [];

        foreach($data as $p) {
            $ids[] = (int)$p["id"];
            $items[(int)$p["id"]] = $this->convertItem($store, $taxCalculator, $p);
        }

        $stockCollection->addProductsFilter($ids);
        $stockCollection->joinStockStatus($store->getId());

        foreach($stockCollection as $stockItem) {
            $productId = (int)$stockItem->getProductId();

            if( ! array_key_exists($productId, $items)) {
                continue;
            }

            if( ! $stockItem->getManageStock()) {
                continue;
            }

            $items[$productId]["stock"] = [
                "inStock" => (bool)$stockItem->getIsInStock(),
            ];
        }


        return [200, array_values($items)];
    }

    public function getItem(string $id): array {
        $store = Mage::app()->getStore();
        $productResource = Mage::getModel("awardit_orderapi/product");
        $stockCollection = Mage::getResourceModel("cataloginventory/stock_item_collection");
        $taxCalculator = Mage::getSingleton("tax/calculation");

        $data = $productResource->getProduct($store, $id);

        if( ! $data) {
            return [404];
        }

        $stockCollection->addProductsFilter([$data["id"]]);
        $stockCollection->joinStockStatus($store->getId());

        $item = $this->convertItem($store, $taxCalculator, $data);

        foreach($stockCollection as $stockItem) {
            if( ! $stockItem->getManageStock()) {
                continue;
            }

            $item["stock"] = [
                "inStock" => (bool)$stockItem->getIsInStock(),
            ];
        }

        return [200, $item];
    }

    private function getMediaUrl(Mage_Core_Model_Store $store, string $image): string {
        $mediaBaseUrl = (string)$store->getConfig("web/secure/base_media_url");

        return $mediaBaseUrl . implode("/", ["catalog", "product", trim($image, "/")]);
    }

    /**
     * @param Product $p
     */
    private function convertItem(
        Mage_Core_Model_Store $store,
        Mage_Tax_Model_Calculation $taxCalculator,
        array $p
    ): array {
        $countryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);
        $customerTaxClass = $taxCalculator->getDefaultCustomerTaxClass($store);

        return [
            "sku" => $p["sku"],
            "name" => $p["name"],
            "description" => $p["description"],
            "image" => $p["image"] ? [ "url" => $this->getMediaUrl($store, $p["image"]) ] : null,
            "price" => [
                "value" => (float)$p["price"],
                "vatPercent" => $taxCalculator->getRate(new Varien_Object([
                    "country_id" => $countryId,
                    "customer_class_id" => $customerTaxClass,
                    "product_class_id" => $p["tax_class_id"],
                ])),
            ],
            "virtual" => $p["type"] === "virtual",
            "enabled" => $p["status"] == Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "categories" => $this->convertProductCategories($p["categories"]),
            "stock" => null,
        ];
    }

    /**
     * @return list<array{name:string}>
     */
    private function convertProductCategories(?string $categories): array {
        if(empty($categories)) {
            return [];
        }

        return array_values(array_filter(array_map(function(string $categoryName): ?array {
            $categoryName = trim($categoryName);

            if( ! $categoryName) {
                return null;
            }

            return [
                "name" => $categoryName,
            ];
        }, explode(Awardit_OrderApi_Model_Product::SEPARATOR, $categories))));
    }
}
