<?php

declare(strict_types=1);

use Awardit\OrderApi\Extension\Quote;
use Awardit\OrderApi\Extension\Order;

class Awardit_OrderApi_OrdersController extends Awardit_OrderApi_Controller_Base {
    const CONFIG_ALLOWED_COUNTRIES = "general/country/allow";
    const ADDRESS_KEYS = [
        "firstname" => true,
        "lastname" => true,
        "street" => true,
        "postcode" => true,
        "city" => true,
        "country_id" => true,
        "company" => true,
        "region" => true,
        "telephone" => true,
    ];

    public function getAll(): array {
        return [404];
    }

    public function getItem(string $id): array {
        $store = Mage::app()->getStore();
        /**
         * @var Order
         */
        $order = Mage::getModel("sales/order");

        $order->loadByIncrementId($id);

        if( ! $order->getId() || $order->getStoreId() != $store->getId()) {
            return [404];
        }

        return [200, $this->formatOrder($order)];
    }

    /**
     * @param Order $order
     */
    private function formatOrder(Mage_Sales_Model_Order $order): array {
        $status = $this->getOrderStatus($order);

        // FIXME: Read from magento
        $parcels = [];

        return [
            "orderId" => $order->getIncrementId(),
            "status" => $status,
            "parcels" => $parcels,
        ];
    }

    private function getOrderStatus(Mage_Sales_Model_Order $order): string {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $select = $connection->select();

        $select->from(["o" => Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS]);
        $select->where("o.magento_order_id = ?", $order->getId());
        $select->limit(1);

        $data = array_values($connection->fetchAll($select))[0] ?? null;
        $status = $data ? ($data["export_status"] ?? 0) : 0;

        switch($status) {
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_NEW: // New
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_PREPARED: // Prepared (XML-file created and saved)
            return "pending";
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_TRANSFERRED1: // Transferred Step 1 (XML-file transferred to Visma)
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_TRANSFERRED2: // Transferred Step 2 (XML-file transferred to Visma)
            return "accepted";
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_VERIFIED: // Verified (Have visma_order_id and visma_status)
            return "processing";
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_SHIPPED: // Shipped (Order status (4) in Visma indicate order has shipped)
            return "shipped";
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_COMPLETE: // Order complete
            return "complete";
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_CANCELED: // Order has been canceled (Order status (6) in Visma)
        case Awardit_Integration_Model_Cli_OrderExport::EXPORT_STATUS_ERROR: // Order have some kind of error
            return "canceled";

        default:
            return "pending";
        }
    }

    public function createItem(): array {
        $data = $this->requestData();
        /** @var Quote */
        $quote = Mage::getModel("sales/quote");
        $helper = Mage::helper("awardit_orderapi");
        $store = Mage::app()->getStore();

        if( ! $data || ! is_array($data)) {
            throw new Awardit_OrderApi_JsonParseException();
        }

        if( ! array_key_exists("items", $data) ||
            ! is_array($data["items"]) ||
            ! array_is_list($data["items"])
        ) {
            throw new Awardit_OrderApi_BadRequestException("Missing items list");
        }

        foreach($data["items"] as $k => $item) {
            $this->addItem($quote, $item, $k);
        }

        if( ! array_key_exists("recipient", $data) ||
            ! is_array($data["recipient"])
        ) {
            throw new Awardit_OrderApi_BadRequestException("Missing recipient object");
        }

        $this->setRecipient($quote, $data["recipient"]);

        if(array_key_exists("externalReference", $data) && $data["externalReference"]) {
            if( ! is_string($data["externalReference"])) {
                throw new Awardit_OrderApi_BadRequestException("Property externalReference must be string or null");
            }

            $quote->setExternalOrderApiReference($data["externalReference"]);
        }

        if( ! $quote->isVirtual()) {
            $shippingAddress = $quote->getShippingAddress();
            $billingAddress = $quote->getBillingAddress();

            $billingAddress->addData(array_intersect_key($shippingAddress->getData(), self::ADDRESS_KEYS));

            $quote->setTotalsCollectedFlag(false);
            $shippingAddress->setSameAsBilling(true);
            $shippingAddress->setCollectShippingRates(true);

            $quote->collectTotals();
            $shippingAddress->collectShippingRates();

            $shippingMethod = $helper->getShippingMethod($store);

            if( ! $shippingMethod) {
                throw new Exception("Missing shipping method in configuration");
            }

            Mage::log("Attempting to use shipping method '$shippingMethod'", Zend_Log::DEBUG, "externalorderapi");

            $shippingAddress->setShippingMethod($shippingMethod);
            $shippingAddress->setCollectShippingRates(true);
        }

        $payment = $quote->getPayment();

        $payment->importData([
            // FIXME: Configurable, or use a custom payment method
            "method" => "checkmo"
        ]);

        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        $service = Mage::getModel("sales/service_quote", $quote);

        try {
            $service->submitAll();

            /**
             * @var Order
             */
            $order = $service->getOrder();

            // Manually set order state since checkmo does not allow processing
            $order->setState(Mage_Sales_Model_Order::STATE_PROCESSING, true)->save();

            if($order->getCanSendNewEmailFlag()) {
                try {
                    $order->queueNewOrderEmail();
                }
                catch(Exception $e) {
                    Mage::logException($e);
                }
            }

            try {
                // Send any virtual products since we have an authorization approved
                Mage::dispatchEvent('crossroads_order_payment_complete', [ "order" => $order ]);
            }
            catch(Exception $e) {
                Mage::logException($e);
            }

            return [201, $this->formatOrder($order)];
        }
        catch(Mage_Core_Exception $e) {
            // Convert known Mage_Core_Exception instances which normally happen
            // and are user-visible
            switch($e->getMessage()) {
            case Mage::helper("cataloginventory")
                ->__("Not all products are available in the requested quantity"):
                throw new Awardit_OrderApi_BadRequestException("Item not in stock");
            default:
                throw $e;
            }
        }
        finally {
            // Always deactivate, we do not have any sessions
            $quote->setIsActive(0);
            $quote->save();
        }
    }

    /**
     * @param Array<mixed> $data
     */
    private function setRecipient(Mage_Sales_Model_Quote $quote, array $data): void {
        if( ! array_key_exists("email", $data) ||
            ! is_string($data["email"]) ||
            empty($data["email"])
        ) {
            throw new Awardit_OrderApi_BadRequestException("Missing recipient.email");
        }

        $email = $data["email"];

        if( ! Zend_Validate::is($email, "EmailAddress")) {
            throw new Awardit_OrderApi_BadRequestException("Invalid email in recipient.email");
        }

        $quote->setCustomerEmail($email);

        if( ! array_key_exists("telephone", $data) ||
            ! is_string($data["telephone"]) ||
            empty($data["telephone"])
        ) {
            throw new Awardit_OrderApi_BadRequestException("Missing recipient.telephone");
        }

        $telephone = $data["telephone"];

        if( ! $quote->isVirtual()) {
            if( ! array_key_exists("address", $data) ||
                ! is_array($data["address"])
            ) {
                throw new Awardit_OrderApi_BadRequestException("Property recipient.address is required when order contains physical products");
            }
        }

        $address = $quote->isVirtual() ? $quote->getBillingAddress() : $quote->getShippingAddress();

        if(array_key_exists("address", $data) && $data["address"]) {
            if( ! is_array($data["address"])) {
                throw new Awardit_OrderApi_BadRequestException("Property recipient.address must be an object");
            }

            $this->setAddress($address, $data["address"]);
        }
        else {
            $this->setAddressPlaceholder($address);
        }

        $address->setEmail($email);
        $address->setTelephone($telephone);
    }

    /**
     * @param mixed $value
     * @psalm-assert non-empty-string $value
     */
    private function assertNonEmptyString(string $property, $value): void {
        if( ! $value || ! is_string($value)) {
            throw new Awardit_OrderApi_BadRequestException("Property recipient.address.$property must be a string");
        }
    }

    /**
     * @param Array<mixed> $data
     */
    private function setAddress(Mage_Sales_Model_Quote_Address $address, $data): void {
        $company = $data["company"] ?? null;
        $firstname = $data["firstname"] ?? null;
        $lastname = $data["lastname"] ?? null;
        $street = $data["street"] ?? null;
        $postcode = $data["postcode"] ?? null;
        $city = $data["city"] ?? null;
        $region = $data["region"] ?? null;
        $countryId = $data["countryId"] ?? null;

        $this->assertNonEmptyString("firstname", $firstname);
        $this->assertNonEmptyString("lastname", $lastname);
        $this->assertNonEmptyString("postcode", $postcode);
        $this->assertNonEmptyString("city", $city);
        $this->assertNonEmptyString("countryId", $countryId);

        if( ! in_array($countryId, $this->getAllowedCountryCodes(Mage::app()->getStore()))) {
            throw new Awardit_OrderApi_BadRequestException("Property recipient.address.countryId must be a valid country id");
        }

        if( ! is_array($street) || ! array_is_list($street)) {
            throw new Awardit_OrderApi_BadRequestException("Property recipient.address.street must be a list of strings");
        }

        foreach($street as $v) {
            if( ! is_string($v)) {
                throw new Awardit_OrderApi_BadRequestException("Property recipient.address.street must be a list of strings");
            }
        }

        /** @var list<string> $street */

        $address->setFirstname($firstname);
        $address->setLastname($lastname);
        $address->setStreet($street);
        $address->setPostcode($postcode);
        $address->setCity($city);
        $address->setCountryId($countryId);

        if($company) {
            if( ! is_string($company)) {
                throw new Awardit_OrderApi_BadRequestException("Property recipient.address.company must be a string or null");
            }

            $address->setCompany($company);
        }

        if($region) {
            if( ! is_string($region)) {
                throw new Awardit_OrderApi_BadRequestException("Property recipient.address.region must be a string or null");
            }

            $address->setRegion($region);
        }
    }

    private function setAddressPlaceholder(Mage_Sales_Model_Quote_Address $address): void {
        $store = Mage::app()->getStore();
        $defaultCountryId = (string)$store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

        $address->setFirstname("-");
        $address->setLastname("-");
        $address->setStreet(["-"]);
        $address->setPostcode("-");
        $address->setCity("-");
        $address->setCountryId($defaultCountryId);
    }

    /**
     * @param mixed $item
     */
    private function addItem(Mage_Sales_Model_Quote $quote, $item, int $index): void {
        $store = Mage::app()->getStore();

        if( ! $item || ! is_array($item)) {
            throw new Awardit_OrderApi_BadRequestException("items.$index is not an object");
        }

        if( ! array_key_exists("qty", $item) || ! is_int($item["qty"])) {
            throw new Awardit_OrderApi_BadRequestException("items.$index.qty is not an integer");
        }

        if( ! array_key_exists("sku", $item) || ! is_string($item["sku"])) {
            throw new Awardit_OrderApi_BadRequestException("items.$index.sku is not a string");
        }

        $sku = $item["sku"];

        $product = Mage::getModel("catalog/product");
        $productId = $product->getIdBySku($sku);

        if( ! $productId) {
            throw new Awardit_OrderApi_ProductNotFoundException($sku);
        }

        $product->setStoreId($store->getId());
        $product->load($productId);

        if( ! Mage::helper("awardit_orderapi")->isProductVisible($product, $store)) {
            throw new Awardit_OrderApi_ProductNotFoundException($sku);
        }

        try {
            $quote->addProduct($product, new Varien_Object([]));
        }
        catch(Exception $e) {
            Mage::logException($e);

            throw $e;
        }
    }

    private function isValidCountryId(
        ?string $countryId,
        Mage_Core_Model_Store $store
    ): bool {
        if(!$countryId) {
            return false;
        }

        $approvedIds = $this->getAllowedCountryCodes($store);

        return in_array($countryId, $approvedIds);
    }

    /**
     * @return Array<string>
     */
    private function getAllowedCountryCodes(Mage_Core_Model_Store $store): array {
        return array_map("strtoupper", array_filter(explode(",", $store->getConfig(self::CONFIG_ALLOWED_COUNTRIES) ?: "")));
    }
}
