<?php

declare(strict_types=1);

/**
 * @psalm-type Product array{id:string, sku:string, type:string, name:string, description:string, image:string, price:string, status:string, tax_class_id:string, categories:string}
 */
class Awardit_OrderApi_Model_Product {
    const SEPARATOR = "±";

    /**
     * @param array{limit?:mixed, offset?:mixed} $pagination
     * @return list<Product>
     */
    public function getProducts(Mage_Core_Model_Store $store, array $pagination): array {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $select = $this->select($store, $connection);

        $select->order("p.entity_id asc");

        $limit = (int)($pagination["limit"] ?? 0);
        $offset = (int)($pagination["offset"] ?? 0);

        $select->limit(
            ($limit > 0 && $limit <= 500) ? $limit : 100,
            $offset >= 0 ? $offset : 0
        );

        /** @var list<Product> */
        return array_values($connection->fetchAll($select));
    }

    /**
     * @return ?Product
     */
    public function getProduct(Mage_Core_Model_Store $store, string $sku): ?array {
        $resource = Mage::getSingleton("core/resource");
        $connection = $resource->getConnection("core_read");

        assert($connection !== false);

        $select = $this->select($store, $connection);

        $select->where("p.sku = ?", $sku);
        $select->limit(1);

        /** @var ?Product */
        return array_values($connection->fetchAll($select))[0] ?? null;
    }

    private function select(
        Mage_Core_Model_Store $store,
        Varien_Db_Adapter_Interface $connection
    ): Zend_Db_Select {
        $select = $connection->select();
        $resource = Mage::getSingleton("core/resource");
        $websiteId = (int)$store->getWebsiteId();

        $select->from([
            "p" => $resource->getTableName("catalog/product"),
        ], [
            "id" => "entity_id",
            "sku" => "sku",
            "type" => "type_id"
        ]);

        // Only show products assigned to the website
        $select->join([
            "w" => $resource->getTableName("catalog/product_website")
        ], "w.product_id = p.entity_id AND w.website_id = {$websiteId}", []);

        $this->joinAttribute($store, $select, "name");
        $this->joinAttribute($store, $select, "description");
        $this->joinAttribute($store, $select, "image");
        $this->joinAttribute($store, $select, "price");
        $this->joinAttribute($store, $select, "status");
        $this->joinAttribute($store, $select, "tax_class_id");

        $categories = $this->selectCategoryNames($store, $connection);

        $categories->join([
            "cp" => $resource->getTableName("catalog/category_product"),
        ], "c.entity_id = cp.category_id", []);
        $categories->where("cp.product_id = p.entity_id");

        $select->columns([
            "categories" => $categories,
        ]);

        $select->where("p.type_id IN (?)", ["simple", "virtual"]);
        $select->where("p.has_options = 0");

        return $select;
    }

    private function selectCategoryNames(
        Mage_Core_Model_Store $store,
        Varien_Db_Adapter_Interface $connection
    ): Zend_Db_Select {
        $config = Mage::getSingleton("eav/config");
        $resource = Mage::getSingleton("core/resource");
        $storeId = (int)$store->getId();
        $attr = $config->getAttribute("catalog_category", "name");

        assert($attr !== false);

        $attrId = (int)$attr->getId();
        /** @var string */
        $backend = $attr->getBackendType();
        $table = $resource->getTableName("catalog/category")."_".$backend;

        $select = $connection->select();

        $select->from([
            "c" => $resource->getTableName("catalog/category"),
        ], []);

        $select->joinLeft(["csv" => $table], "csv.store_id = {$storeId} AND csv.entity_id = p.entity_id AND csv.attribute_id = {$attrId}", []);
        $select->joinLeft(["cgv" => $table], "cgv.store_id = 0 AND cgv.entity_id = p.entity_id AND cgv.attribute_id = {$attrId}", []);

        $select->columns([
            "categories" => new Zend_Db_Expr("GROUP_CONCAT(COALESCE(csv.value, cgv.value) SEPARATOR '".self::SEPARATOR."')"),
        ]);

        return $select;
    }

    private function joinAttribute(
        Mage_Core_Model_Store $store,
        Zend_Db_Select $select,
        string $attributeCode
    ): void {
        $config = Mage::getSingleton("eav/config");
        $resource = Mage::getSingleton("core/resource");
        $storeId = (int)$store->getId();
        $attr = $config->getAttribute("catalog_product", $attributeCode);

        assert($attr !== false);

        $attrId = (int)$attr->getId();
        /** @var string */
        $backend = $attr->getBackendType();
        $table = $resource->getTableName("catalog/product")."_".$backend;

        $select->columns([
            $attributeCode => new Zend_Db_Expr("COALESCE({$attributeCode}.value, {$attributeCode}_g.value, '')"),
        ]);
        $select->joinLeft([$attributeCode => $table], "{$attributeCode}.store_id = {$storeId} AND {$attributeCode}.entity_id = p.entity_id AND {$attributeCode}.attribute_id = {$attrId}", []);
        $select->joinLeft([$attributeCode."_g" => $table], "{$attributeCode}_g.store_id = 0 AND {$attributeCode}_g.entity_id = p.entity_id AND {$attributeCode}_g.attribute_id = {$attrId}", []);
    }
}
