<?php

declare(strict_types=1);

namespace Fixtures;

use Crossroads\Magento\Test\Integration\MagentoManager;
use Exception;
use Mage;
use Mage_Catalog_Model_Product_Type_Configurable;
use Mage_Bundle_Model_Product_Type;
use Mage_Catalog_Model_Category;
use Mage_Catalog_Model_Entity_Attribute;
use Mage_Catalog_Model_Product;
use Mage_Catalog_Model_Product_Option;
use Mage_Catalog_Model_Product_Status;
use Mage_Catalog_Model_Product_Visibility;
use Mage_Catalog_Model_Resource_Eav_Attribute;
use Mage_Catalog_Model_Resource_Setup;
use Mage_Core_Model_Store;
use Mage_Eav_Model_Entity_Setup;

/**
 * @psalm-import-type BundleOptions from Products
 */
class OrderProducts extends Products {
    public function load(): void {
        parent::load();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        $this->createVirtualBundle();

        MagentoManager::reindex();
        MagentoManager::reset();
    }

    /**
     * A product with a single optional option which is enabled by default and is virtual.
     */
    public function createVirtualBundle(): void {
        /** @var Mage_Core_Model_Store */
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        // We reuse the virtual product here
        $optionModel = Mage::getModel("catalog/product");

        $optionModel->load($optionModel->getIdBySku("test-virtual"));

        $model = $this->loadOrCreate([
            "website_ids" => [$store->getWebsiteId()],
            "type_id" => "bundle",
            "sku" => "test-bundle-virtual",
            "name" => "Test Bundle Virtual Default Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class_id" => TaxRates::getTaxClass(TaxRates::PRODUCT_TAX_CLASS)->getId(),
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 0,
            "price_type" => 0, // Dynamic (0), Fixed (1)
            "price_view" => 0, // Price range (0), Price as low as (1)
            "weight_type" => 0, // Dynamic (0), Fixed (1)
            "shipment_type" => 0, // Together (0), Separately (1)
            "meta_title" => "Test Bundle Virtual Default meta title",
            "meta_description" => "Test Bundle meta description",
            "meta_keyword" => "bundle,meta,keywords",
            "description" => "The long Test Bundle Virtual Default description",
            "short_description" => "The shortest Test Bundle Virtual Default description",
            "small_image" => "product_bundle_small_image.jpg",
            "small_image_label" => "Small Bundle Image",
            "thumbnail" => "product_bundle_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Bundle Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
        ]);

        $this->assignCategory(
            $model,
            MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category"),
            4
        );

        Mage::getModel("cataloginventory/stock_item")->addData([
            "stock_id" => 1,
            "use_config_manage_stock" => 1,
            "manage_stock" => 1,
            "is_in_stock" => 1,
        ])->assignProduct($model)->save();

        $this->assignBundleOptions($model, [
            [
                "title" => "Include extra",
                "type" => "checkbox",
                "required" => 1, // Required
                "position" => 1,
                "selections" => [
                    [
                        "product_id" => (string)$optionModel->getId(),
                        "selection_price_value" => 0,
                        "selection_price_type" => 0,
                        "selection_qty" => 2,
                        "selection_can_change_qty" => 0,
                        "is_default" => 1, // But is set to default
                        "position" => 1,
                    ],
                ],
            ],
        ]);
    }

    /**
     * @param BundleOptions $bundleData
     */
    public function assignBundleOptions(
        Mage_Catalog_Model_Product $product,
        array $bundleData
    ): void {
        $bundleOptions = [];
        $bundleSelections = [];
        /**
         * @var Mage_Bundle_Model_Product_Type
         */
        $instance = $product->getTypeInstance(true);
        $optionColl = $instance->getOptionsCollection($product);

        foreach($optionColl as $o) {
            foreach($bundleData as $d) {
                if($o->getDefaultTitle() == $d["title"]) {
                    continue 2;
                }
            }

            $o->delete();
        }

        foreach($bundleData as $k => $d) {
            $option = null;

            foreach($optionColl as $o) {
                if($o->getDefaultTitle() == $d["title"]) {
                    $option = [
                        "title" => $o->getDefaultTitle(),
                        "option_id" => $o->getId(),
                        "type" => $d["type"],
                        "required" => $d["required"],
                        "position" => $d["position"] ?? 0,
                    ];

                    break;
                }
            }

            if( ! $option) {
                $option = $d;
            }

            $option["delete"] = "";
            $option["option_id"] = $option["option_id"] ?? 0;

            if($option["option_id"]) {
                $k = $option["option_id"];
            }

            $bundleOptions[$k] = $option;

            $selColl = isset($option["option_id"]) ?
                Mage::getResourceModel("bundle/selection_collection")
                    ->setOptionIdsFilter([$option["option_id"]]) :
                [];

            foreach($selColl as $os) {
                foreach($d["selections"] as $ds) {
                    if($ds["product_id"] == $os->getProductId()) {
                        continue 2;
                    }
                }

                $os->delete();
            }

            foreach($d["selections"] as $ds) {
                $sel = null;

                foreach($selColl as $os) {
                    if($os->getProductId() == $ds["product_id"]) {
                        // The selection collection contains products, not selections
                        $sel = [
                            "selection_id" => (int)$os->getSelectionId(),
                            "product_id" => (int)$os->getProductId(),
                            "selection_price_value" => $ds["selection_price_value"],
                            "selection_price_type" => $ds["selection_price_type"],
                            "selection_qty" => $ds["selection_qty"],
                            "selection_can_change_qty" => $ds["selection_can_change_qty"],
                            "is_default" => $ds["is_default"],
                            "position" => $ds["position"] ?? 0,
                        ];

                        break;
                    }
                }

                if( ! $sel) {
                    $sel = $ds;
                };

                $sel["delete"] = "";

                if( ! array_key_exists($k, $bundleSelections)) {
                    $bundleSelections[$k] = [];
                }

                $bundleSelections[$k][] = $sel;
            }
        }

        // Got to enable flags and register to be able to properly save a bundle product
        $product->setCanSaveCustomOptions(true);
        $product->setCanSaveBundleSelections(true);

        Mage::register("product", $product);

        $product->setBundleOptionsData($bundleOptions);
        $product->setBundleSelectionsData($bundleSelections);

        $product->save();

        Mage::unregister("product");
    }
}
