<?php

declare(strict_types=1);

/**
 * Base controller for REST plural resources.
 *
 * @psalm-type Response array{0:int, 1?:?array}
 */
abstract class Awardit_OrderApi_Controller_Base extends Mage_Core_Controller_Front_Action {
    const LOG_CHANNEL = "externalorderapi";

    /**
     * List all items in this resource, applying filters from the request, reached using `GET /resource`.
     *
     * @return Response
     */
    public abstract function getAll(): array;

    /**
     * Displays a single item from the resource, reached using `GET /resource/:id`.
     *
     * @return Response
     */
    public abstract function getItem(string $id): array;

    /**
     * Replaces all items in this resource with the supplied data from the client,
     * reached using `PUT /resource`.
     *
     * @return Response
     */
    public function replaceAll(): array {
        return [405];
    }

    /**
     * Patches the items in this resource with the supplied change-set from the client,
     * reached using `PATCH /resource`.
     *
     * @return Response
     */
    public function updateAll(): array {
        return [405];
    }

    /**
     * Empties this resource, reached using `DELETE /resource`.
     *
     * @return Response
     */
    public function deleteAll(): array {
        return [405];
    }

    /**
     * Adds an item to the resource, reached using `POST /resource`.
     *
     * @return Response
     */
    public function createItem(): array {
        return [405];
    }

    /**
     * Updates an item in the collection by replacing its data, reached using `PUT /resource/:id`.
     *
     * @psalm-suppress PossiblyUnusedParam
     * @return Response
     */
    public function replaceItem(string $id): array {
        return [405];
    }

    /**
     * Patches an item in this collection with the supplied change-set from the client,
     * reached using `PATCH /resource/:id`.
     *
     * @psalm-suppress PossiblyUnusedParam
     * @return Response
     */
    public function updateItem(string $id): array {
        return [405];
    }

    /**
     * Deletes an item in the collection, reached using `DELETE /resource/:id`.
     *
     * @psalm-suppress PossiblyUnusedParam
     * @return Response
     */
    public function deleteItem(string $id): array {
        return [405];
    }

    /**
     * Adds an eleemnt to a subresource, reached using `POST /resource/:id`.
     *
     * @psalm-suppress PossiblyUnusedParam
     * @return Response
     */
    public function addToItem(string $id): array {
        return [405];
    }

    // Override base action
    public function indexAction(): void {
        try {
            /** @var Mage_Core_Model_Store */
            $store = Mage::app()->getStore();
            $helper = Mage::helper("awardit_orderapi");

            $helper->ensureAccess($store, $this->getRequest());

            switch($this->getRequest()->getMethod()) {
                case "GET":
                    $this->sendData($this->getAll());
                    return;
                case "POST":
                    $this->sendData($this->createItem());
                    return;
                case "PUT":
                    $this->sendData($this->replaceAll());
                    return;
                case "PATCH":
                    $this->sendData($this->updateAll());
                    return;
                case "DELETE":
                    $this->sendData($this->deleteAll());
                    return;
                default:
                    $this->sendData([405]);
                    return;
            }
        }
        catch(Exception $e) {
            $this->handleException($e);
        }
    }

    // Using true here we tell Magento's router that we support all actions.
    public function hasAction($action) { return true; }

    // Catch all for all items in this resource
    public function norouteAction($coreRoute = null): void {
        $request = $this->getRequest();
        // api/_controller_/_id_
        $uri = array_map("rawurldecode", explode("/", trim($request->getPathInfo(), "/")));
        $id = current(array_slice($uri, 2, 1));

        if( ! $id) {
            $this->indexAction();
            return;
        }

        try {
            /** @var Mage_Core_Model_Store */
            $store = Mage::app()->getStore();
            $helper = Mage::helper("awardit_orderapi");

            $helper->ensureAccess($store, $this->getRequest());

            switch($request->getMethod()) {
                case "GET":
                    $this->sendData($this->getItem($id));
                    return;
                case "POST":
                    $this->sendData($this->addToItem($id));
                    return;
                case "PUT":
                    $this->sendData($this->replaceItem($id));
                    return;
                case "PATCH":
                    $this->sendData($this->updateItem($id));
                    return;
                case "DELETE":
                    $this->sendData($this->deleteItem($id));
                    return;
                default:
                    $this->sendData([405]);
            }
        }
        catch(Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * For outputting a JSON response
     *
     * @param Response $data
     */
    protected function sendData(array $data): void {
        if(!$data[0]) {
            throw new Exception("Missing return array.");
        }

        $this->getResponse()->setHttpResponseCode($data[0]);

        if(array_key_exists(1, $data)) {
            $this->getResponse()
                ->setHeader("Content-Type", "application/json; charset=utf-8", true)
                ->setBody(json_encode($data[1], JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES));
        }
    }

    /**
     * Get JSON request data as an assoc array, will output null if Content-Type
     * is not application/json.
     *
     * NOTE: Does not accept NULL as a json string.
     *
     * @return ?mixed attached data, null if not application/json
     */
    protected function requestData() {
        $request = $this->getRequest();
        $content_type = $request->getHeader("Content-Type") ?: "";

        if(strtolower(trim($content_type)) === "application/json") {
            /** @var mixed */
            $data = json_decode($request->getRawBody() ?: "", true);

            if($data === null) {
                throw new Awardit_OrderApi_JsonParseException();
            }

            return $data;
        }

        return null;
    }

    private function handleException(Throwable $t): void {
        $response = $this->getResponse();

        if( ! $t instanceof Awardit_OrderApi_ResponseException) {
            Mage::logException($t, self::LOG_CHANNEL);

            if(Mage::getIsDeveloperMode()) {
                throw $t;
            }

            $response->setHttpResponseCode(500);

            return;
        }

        Mage::logException($t, self::LOG_CHANNEL, $t->getErrorLevel());

        $response->setHttpResponseCode($t->getHttpResponseCode());

        foreach($t->getHttpHeaders() as $k => $v) {
            $response->setHeader($k, $v, true);
        }

        $body = $t->getBody();

        if($body) {
            $response->setBody($body);
        }
    }
}
