<?php

declare(strict_types=1);

/* Methods from SimpleXMLElement and Varien_Simplexml_Element:
 *
 * ```
 * attributes(): ?SimpleXMLElement
 * children(): ?SimpleXMLElement
 * count(): int
 * getName(): string
 * __toString(): string
 *
 * // Not implemented:
 * asXML(?string $filename = null): string|bool
 *
 * // Not implemented since we do not deal with namespaces:
 * getDocNamespaces(bool $recursive = false, bool $fromRoot = true): array|false
 * getNamespaces(bool $recursive = false): array
 * registerXPathNamespace(string $prefix, string $namespace): bool
 *
 * // Not implemented since the object is read-only:
 * addAttribute(string $qualifiedName, string $value): void
 * addChild(string $qualifiedName, ?string $value = null): ?SimpleXMLElement
 *
 * // The following methods from the original Varien_Simplexml_Element class are
 * // implemented:
 * getAttribute(string $name): ?string
 * hasChildren(): bool
 * asArray(): array
 * asCanonicalArray(): array
 * descend(string $path): Varien_Simplexml_Element|false
 *
 * // The following methods from the original Varien_Simplexml_Element class are
 * // not implemented:
 * getParent(): Varien_Simplexml_Element|false
 * setParent(Varien_Simplexml_Element $element): void
 * asNiceXml(string $filename = "", int|false $level = 0): string
 * innerXml(int|false $level = 0): string
 * xmlentities(mixed $value = null): string
 *
 * // The following methods for modifying the original Varien_Simplexml_Element
 * // class are not implemented, see Varien_Simplexml_Config for ways to modify
 * // the configuration:
 * appendChild(Varien_Simplexml_Element $source): Varien_Simplexml_Element
 * extend(Varien_Simplexml_Element $source, bool $overwrite = false): Varien_Simplexml_Element
 * extendChild(Varien_Simplexml_Element $source, bool $overwrite = false): Varien_Simplexml_Element
 * setNode(string $path, ?string $value, bool $overwrite): void
 * ```
 */

/**
 * Object emulating SimpleXMLElement but with plain PHP data inside.
 *
 * @psalm-import-type Varien_Simplexml_Node from Varien_Simplexml_Util
 * @psalm-import-type Varien_Simplexml_LeafNode from Varien_Simplexml_Util
 *
 * @implements ArrayAccess<string, string>
 * @implements IteratorAggregate<string, Varien_Simplexml_Element&static>
 */
class Varien_Simplexml_Element implements ArrayAccess, Countable, IteratorAggregate, Stringable {
    final const ITER_CHILDREN = "ITER_CHILDREN";
    final const ITER_SELF = "ITER_SELF";

    /**
     * Key where leaf values are stored for elements which also have attributes.
     */
    final const VALUE_KEY = "#";
    /**
     * Key where attribute values are stored inside elements.
     *
     * Attributes in asArray are stored in @, so we duplicate this by
     * construction here.
     */
    final const ATTRIBUTES_KEY = "@";

    // Constructor is final since we will create child-objects as needed
    // through this.
    /**
     * @param list<Varien_Simplexml_Node|Varien_Simplexml_LeafNode> $nodes
     * @param self::ITER_* $iteratorType
     */
    final public function __construct(
        private readonly string $name,
        /**
         * List of nodes at this nesting with the given name
         */
        private readonly array $nodes,
        /**
         * If this node is the root node, iteration over a root node iterates children.
         */
        private readonly string $iteratorType
    ) {}

    /**
     * Emulate object property getters to traverse the tree, nodes reached
     * this way will iterate over themselves instead of their children.
     */
    // Shows up a decent amount in profiling traces, but those are non-sampling
    // and will overrepresent small methods which are called often. Probably
    // not worth optimizing, and the JIT should be good for methods like these.
    public function __get(string $name): ?static {
        if( ! isset($this->nodes[0][$name])) {
            // This differs from SimpleXMLElement, first element which does not
            // exist results in an empty SimpleXMLElement, but next attempt
            // results in null. Null right away instead to make it clear:
            return null;
        }

        /**
         * We never access it using ->{"@"} or ->{"#"}
         */
        // Just grab the first one, the fact that we have a key means we have
        // child nodes.
        return new static($name, $this->nodes[0][$name], self::ITER_SELF);
    }

    /**
     * Same logic as __get, but for isset.
     */
    public function __isset(string $name): bool {
        return isset($this->nodes[0][$name]);
    }

    /**
     * Attempts to set a property will throw since all the element objects are
     * read-only, instead modify the configuration via the root
     * Varien_Simplexml_Config object it belongs to.
     */
    public function __set(string $name, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set property %s::%s", __CLASS__, $name));
    }

    /**
     * Attempts to unset a property will throw since all the element objects are
     * read-only, instead modify the configuration via the root
     * Varien_Simplexml_Config object it belongs to.
     */
    public function __unset(string $name): void {
        throw new RuntimeException(sprintf("Attempting to unset property %s::%s", __CLASS__, $name));
    }

    /**
     * Converts the element to a string if it has a string-value, throws if it
     * is not a leaf-node.
     */
    public function __toString(): string {
        if( ! isset($this->nodes[0][self::VALUE_KEY])) {
            throw new RuntimeException(sprintf("Attempting to convert non-leaf object %s to string", __CLASS__));
        }

        return $this->nodes[0][self::VALUE_KEY];
    }

    /**
     * Integers index nodes, strings index attributes of the first node.
     *
     * @param string|int $offset
     */
    public function offsetExists(mixed $offset): bool {
        if(is_int($offset)) {
            return isset($this->nodes[$offset]);
        }

        return isset($this->nodes[0][self::ATTRIBUTES_KEY][$offset]);
    }

    /**
     * Integers index nodes, strings index attributes of the first node.
     *
     * @param string|int $offset
     * @return static|string|null
     */
    public function offsetGet(mixed $offset): mixed {
        if(is_int($offset)) {
            return isset($this->nodes[$offset]) ?
                new static($this->name, [$this->nodes[$offset]], self::ITER_CHILDREN) :
                null;
        }

        return $this->nodes[0][self::ATTRIBUTES_KEY][$offset] ?? null;
    }

    /**
     * Attempts to set an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetSet(mixed $offset, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set node attribute %s::%s", __CLASS__, (string)$offset));
    }

    /**
     * Attempts to unset an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetUnset(mixed $offset): void {
        throw new RuntimeException(sprintf("Attempting to unset node attribute %s::%s", __CLASS__, (string)$offset));
    }

    /**
     * Iterates over multiple elements on this path.
     *
     * @return Traversable<string, static>
     */
    public function getIterator(): Traversable {
        if($this->iteratorType === self::ITER_CHILDREN) {
            return $this->children();
        }

        $iter = [];

        foreach($this->nodes as $node) {
            $iter[] = [$this->name, [$node]];
        }

        return new Varien_Simplexml_Iterator($iter, $this::class);
    }

    public function count(): int {
        $firstNode = $this->nodes[0] ?? [];
        $count = count($firstNode);

        if(array_key_exists(self::ATTRIBUTES_KEY, $firstNode)) {
            $count--;
        }

        if(array_key_exists(self::VALUE_KEY, $firstNode)) {
            $count--;
        }

        return $count;
    }

    /**
     * Fetches the data-representation of the first element.
     *
     * @internal Varien_Simplexml_*
     * @return Varien_Simplexml_Node|Varien_Simplexml_LeafNode
     */
    public function getInnerData(): array {
        return $this->nodes[0] ?? [];
    }

    /**
     * Returns the node and children as an array.
     *
     * Example structure:
     *
     * [
     *   "@attributes" => [
     *     "attributeKey" => "attributeValue",
     *   ],
     *   "childNodeName" => [
     *     "leafNodeName" => "leafNodeValue",
     *     "@attributes" => [
     *       "theAttribute" => "value",
     *     ],
     *     "attributeNode" => "Value",
     *   ],
     * ]
     */
    public function asArray(): array {
        // We need to flatten the array
        $flat = Varien_Simplexml_Util::flattenKeys($this->nodes[0] ?? []);

        // We might get a string if we are a value-node
        return is_array($flat) ? $flat : [];
    }

    /**
     * Returns the node and children as an array, leaving out any attributes.
     *
     * Example structure:
     *
     * [
     *   "leafNodeNameOne" => "someData",
     *   "childNodeName" => [
     *     "leafNodeName" => "leafNodeValue",
     *   ],
     * ]
     */
    public function asCanonicalArray(): array|string {
        return Varien_Simplexml_Util::flattenKeys($this->nodes[0] ?? [], false);
    }

    /**
     * Returns the name of this node.
     */
    public function getName(): string {
        return $this->name;
    }

    public function getAttribute(string $name): ?string {
        return $this->nodes[0][self::ATTRIBUTES_KEY][$name] ?? null;
    }

    /**
     * Returns a thin object allowing read-only property access to attribute
     * values.
     */
    public function attributes(): Varien_Simplexml_Attributes {
        return new Varien_Simplexml_Attributes($this->nodes[0][self::ATTRIBUTES_KEY] ?? []);
    }

    /**
     * Returns an iterator over children to the first node, excluding attributes.
     *
     * @return Traversable<string, static>
     */
    public function children(): Traversable {
        /**
         * @var Varien_Simplexml_Node|Varien_Simplexml_LeafNode
         */
        $data = $this->nodes[0] ?? [];

        if(array_key_exists(self::VALUE_KEY, $data)) {
            // Empty iterator, we can't iterate on leaf nodes
            return new Varien_Simplexml_Iterator([], $this::class);
        }

        // Skip non-iterable items
        //
        // Unset is slow if the key does not exist (PHP 8.1), which it won't
        // for most elements, which is why we use array_key_exists.
        if(array_key_exists(self::ATTRIBUTES_KEY, $data)) {
            unset($data[self::ATTRIBUTES_KEY]);
        }

        $iter = [];

        // Flatten one level with key
        /**
         * @var Array<string, list<array>> $data
         * @var list<Varien_Simplexml_Node|Varien_Simplexml_LeafNode> $v
        */
        foreach($data as $k => $v) {
            foreach($v as $i) {
                $iter[] = [$k, [$i]];
            }
        }

        return new Varien_Simplexml_Iterator($iter, $this::class);
    }

    /**
     * Returns true if we have non-attribute child nodes.
     */
    public function hasChildren(): bool {
        return $this->count() > 0;
    }

    // TODO: Same logic as Varien_Simplexml_Config::getNode()
    /**
     * @param string|list<string|Stringable> $path
     * @return ?static
     */
    public function descend(string|array $path): ?static {
        $parts = is_array($path) ? array_map("strval", $path) : explode("/", $path);
        $items = $this->nodes;

        foreach($parts as $p) {
            if( ! isset($items[0][$p])) {
                return null;
            }

            /**
             * @var ?list<Varien_Simplexml_Node|Varien_Simplexml_LeafNode>
             */
            $items = $items[0][$p] ?? null;
        }

        return $items !== null ? new static(end($parts) ?: $this->name, $items, self::ITER_CHILDREN) : null;
    }

    public function xpath(string $expression): Traversable {
        return new Varien_Simplexml_Iterator(Varien_Simplexml_Util::xpath($expression, $this->nodes[0] ?? []), $this::class);
    }

    public function serializeData(): string {
        return serialize([
            "name" => $this->name,
            "data" => $this->nodes[0] ?? [],
        ]);
        /*
        return json_encode([
            "name" => $this->name,
            "data" => $this->nodes[0] ?? [],
        ], JSON_PRESERVE_ZERO_FRACTION |
            JSON_THROW_ON_ERROR |
            JSON_UNESCAPED_SLASHES |
            JSON_UNESCAPED_UNICODE);
        */
    }

    /**
     * Converts a leaf-node into an integer, will throw if it is not a leaf-node.
     */
    public function asInt(): int {
        return (int)(string)$this;
    }

    /**
     * Converts a leaf-node into a boolean, will throw if it is not a leaf-node.
     */
    public function asBool(): bool {
        return match(strtolower(trim((string)$this))) {
            "1", "true" => true,
            default => false,
        };
    }
}
