<?php

declare(strict_types=1);

/* Methods from SimpleXMLElement and Varien_Simplexml_Element:
 *
 * ```
 * attributes(): ?SimpleXMLElement
 * children(): ?SimpleXMLElement
 * count(): int
 * getName(): string
 * __toString(): string
 *
 * // Not implemented:
 * asXML(?string $filename = null): string|bool
 *
 * // Not implemented since we do not deal with namespaces:
 * getDocNamespaces(bool $recursive = false, bool $fromRoot = true): array|false
 * getNamespaces(bool $recursive = false): array
 * registerXPathNamespace(string $prefix, string $namespace): bool
 *
 * // Not implemented since the object is read-only:
 * addAttribute(string $qualifiedName, string $value): void
 * addChild(string $qualifiedName, ?string $value = null): ?SimpleXMLElement
 *
 * // The following methods from the original Varien_Simplexml_Element class are
 * // implemented:
 * getAttribute(string $name): ?string
 * hasChildren(): bool
 * asArray(): array
 * asCanonicalArray(): array
 * descend(string $path): Varien_Simplexml_Element|false
 *
 * // The following methods from the original Varien_Simplexml_Element class are
 * // not implemented:
 * getParent(): Varien_Simplexml_Element|false
 * setParent(Varien_Simplexml_Element $element): void
 * asNiceXml(string $filename = "", int|false $level = 0): string
 * innerXml(int|false $level = 0): string
 * xmlentities(mixed $value = null): string
 *
 * // The following methods for modifying the original Varien_Simplexml_Element
 * // class are not implemented, see Varien_Simplexml_Config for ways to modify
 * // the configuration:
 * appendChild(Varien_Simplexml_Element $source): Varien_Simplexml_Element
 * extend(Varien_Simplexml_Element $source, bool $overwrite = false): Varien_Simplexml_Element
 * extendChild(Varien_Simplexml_Element $source, bool $overwrite = false): Varien_Simplexml_Element
 * setNode(string $path, $value
 * ```
 */

/**
 * Object emulating SimpleXMLElement but with plain PHP data inside.
 *
 * @implements ArrayAccess<string, string>
 * @implements IteratorAggregate<string, static>
 */
class Varien_Simplexml_Element implements ArrayAccess, IteratorAggregate, Stringable {
    /**
     * Key where leaf values are stored for elements which also have attributes.
     */
    final const VALUE_KEY = "#";
    /**
     * Key where attribute values are stored inside elements.
     *
     * Attributes in asArray are stored in @, so we duplicate this by
     * construction here.
     */
    final const ATTRIBUTES_KEY = "@";

    // Constructor is final since we will create child-objects as needed
    // through this.
    final public function __construct(
        protected readonly string|false $name,
        protected readonly array $data = []
    ) {}

    /**
     * Emulate object property getters to traverse the tree.
     *
     * @return static|false
     */
    // Shows up a decent amount in profiling traces, but those are non-sampling
    // and will overrepresent small methods which are called often. Probably
    // not worth optimizing, and the JIT should be good for methods like these.
    public function __get(string $name): static|string|false {
        if( ! array_key_exists($name, $this->data)) {
            return false;
        }

        // Just grab the first one, the fact that we have a key means we have
        // child nodes
        return new static($name, $this->data[$name][0]);
    }

    /**
     * Same logic as __get, but for isset.
     */
    public function __isset(string $name): bool {
        return isset($this->data[$name]);
    }

    /**
     * Attempts to set a property will throw since all the element objects are
     * read-only, instead modify the configuration via the root
     * Varien_Simplexml_Config object it belongs to.
     */
    public function __set(string $name, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set property %s::%s", __CLASS__, $name));
    }

    /**
     * Attempts to unset a property will throw since all the element objects are
     * read-only, instead modify the configuration via the root
     * Varien_Simplexml_Config object it belongs to.
     */
    public function __unset(string $name): void {
        throw new RuntimeException(sprintf("Attempting to unset property %s::%s", __CLASS__, $name));
    }

    /**
     * Converts the element to a string if it has a string-value, throws if it
     * is not a leaf-node.
     */
    public function __toString(): string {
        if( ! array_key_exists(self::VALUE_KEY, $this->data)) {
            throw new RuntimeException(sprintf("Attempting to convert non-leaf object %s to string", __CLASS__));
        }

        return $this->data[self::VALUE_KEY];
    }

    /**
     * @param string $offset
     */
    public function offsetExists(mixed $offset): bool {
        return array_key_exists(self::ATTRIBUTES_KEY, $this->data) &&
            array_key_exists($offset, $this->data[self::ATTRIBUTES_KEY]);
    }

    /**
     * @param string $offset
     * @return ?string
     */
    public function offsetGet(mixed $offset): mixed {
        return $this->data[self::ATTRIBUTES_KEY][$offset] ?? null;
    }

    /**
     * Attempts to set an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetSet(mixed $offset, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set node attribute %s::%s", __CLASS__, (string)$offset));
    }

    /**
     * Attempts to unset an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetUnset(mixed $offset): void {
        throw new RuntimeException(sprintf("Attempting to unset node attribute %s::%s", __CLASS__, (string)$offset));
    }

    /**
     * @return Traversable<string, static>
     */
    public function getIterator(): Traversable {
        $data = $this->data;

        if(array_key_exists(self::VALUE_KEY, $data)) {
            // Empty iterator, we can't iterate on leaf nodes
            return new Varien_Simplexml_Iterator([], $this::class);
        }

        // Skip non-iterable items
        //
        // Unset is slow if the key does not exist (PHP 8.1), which it won't
        // for most elements, which is why we use array_key_exists.
        if(array_key_exists(self::ATTRIBUTES_KEY, $data)) {
            unset($data[self::ATTRIBUTES_KEY]);
        }

        $iter = [];

        // Flatten one level with key
        /**
         * @var Array<string, list<array>> $data
         * @var list<array> $v
        */
        foreach($data as $k => $v) {
            foreach($v as $i) {
                $iter[] = [$k, $i];
            }
        }

        return new Varien_Simplexml_Iterator($iter, $this::class);
    }

    // TODO: Is there any way we can make this more ergonomic? Maybe by
    // allowing use of Varien_Simplexml_Element as a part of data to setNode and extendData?
    /**
     * @return Array<string, list<array>>
     */
    public function getInnerData(): array {
        return $this->data;
    }

    /**
     * Returns the node and children as an array.
     *
     * Example structure:
     *
     * [
     *   "@attributes" => [
     *     "attributeKey" => "attributeValue",
     *   ],
     *   "childNodeName" => [
     *     "leafNodeName" => "leafNodeValue",
     *     "@attributes" => [
     *       "theAttribute" => "value",
     *     ],
     *     "attributeNode" => "Value",
     *   ],
     * ]
     */
    public function asArray(): array {
        // We need to flatten the array
        $flat = Varien_Simplexml_Util::flattenKeys($this->data);

        // We might get a string if we are a value-node
        return is_array($flat) ? $flat : [];
    }

    /**
     * Returns the node and children as an array, leaving out any attributes.
     *
     * Example structure:
     *
     * [
     *   "leafNodeNameOne" => "someData",
     *   "childNodeName" => [
     *     "leafNodeName" => "leafNodeValue",
     *   ],
     * ]
     */
    public function asCanonicalArray(): array|string {
        return Varien_Simplexml_Util::flattenKeys($this->data, false);
    }

    /**
     * Returns the name of this node.
     */
    public function getName(): string|false {
        return $this->name;
    }

    public function getAttribute(string $name): ?string {
        return $this->data[self::ATTRIBUTES_KEY][$name] ?? null;
    }

    /**
     * Returns a thin object allowing read-only property access to attribute
     * values.
     */
    public function attributes(): Varien_Simplexml_Attributes {
        return new Varien_Simplexml_Attributes($this->data[self::ATTRIBUTES_KEY] ?? []);
    }

    /**
     * Returns an iterator over children, excluding attributes.
     *
     * @return Traversable<string, static>
     */
    public function children(): Traversable {
        return $this->getIterator();
    }

    /**
     * Returns true if we have non-attribute child nodes.
     */
    public function hasChildren(): bool {
        $count = count($this->data);

        if(array_key_exists(self::ATTRIBUTES_KEY, $this->data)) {
            $count--;
        }

        if(array_key_exists(self::VALUE_KEY, $this->data)) {
            $count--;
        }

        return $count > 0;
    }

    // TODO: Same logic as Varien_Simplexml_Config::getNode()
    public function descend(string $path): Varien_Simplexml_Element|false {
        $parts = is_array($path) ? $path : explode("/", $path);
        $data = $this->data;

        foreach($parts as $p) {
            if( ! is_array($data) || ! array_key_exists((string)$p, $data)) {
                return false;
            }

            $data = $data[$p][0] ?? null;
        }

        return $data !== null ? new static(end($parts), $data) : false;
    }

    public function xpath(string $expression): Traversable {
        return new Varien_Simplexml_Iterator(Varien_Simplexml_Util::xpath($expression, $this->data), $this::class);
    }

    public function getJson(): string {
        return json_encode(
            $this->data,
            JSON_PRESERVE_ZERO_FRACTION |
            JSON_THROW_ON_ERROR |
            JSON_UNESCAPED_SLASHES |
            JSON_UNESCAPED_UNICODE
        );
    }
}
