<?php

declare(strict_types=1);

/**
 * In-memory cache.
 */
class Awardit_Magento_Cache_ProcessMemory extends Zend_Cache_Backend implements Zend_Cache_Backend_Interface {
    /**
     * @var Array<string, array{data:mixed, tags:Array<mixed>}>
     */
    protected $data = [];

    /**
     * Test if a cache is available for the given id and (if yes) return it (false else)
     *
     * Note : return value is always "string" (unserialization is done by the core not by the backend)
     *
     * @param string $id Cache id
     * @param boolean $doNotTestCacheValidity If set to true, the cache validity won't be tested
     * @return string|false cached datas
     */
    public function load($id, $doNotTestCacheValidity = false) {
        if( ! array_key_exists($id, $this->data)) {
            return false;
        }

        return $this->data[$id]["data"];
    }

    /**
     * Test if a cache is available or not (for the given id)
     *
     * @param string $id cache id
     * @return mixed|false (a cache is not available) or "last modified" timestamp (int) of the available cache record
     */
    public function test($id) {
        return array_key_exists($id, $this->data) ? 1 : false;
    }

    /**
     * Save some string datas into a cache record
     *
     * Note : $data is always "string" (serialization is done by the
     * core not by the backend)
     *
     * @param string $data Datas to cache
     * @param string $id Cache id
     * @param array $tags Array of strings, the cache record will be tagged by each string entry
     * @param int|false $specificLifetime If != false, set a specific lifetime for this cache record (null => infinite lifetime)
     * @return boolean true if no problem
     */
    public function save($data, $id, $tags = array(), $specificLifetime = false) {
        $this->data[$id] = [
            "data" => $data,
            "tags" => $tags,
        ];

        return true;
    }

    /**
     * Remove a cache record
     *
     * @param  string $id Cache id
     * @return boolean True if no problem
     */
    public function remove($id) {
        if( ! array_key_exists($id, $this->data)) {
            return false;
        }

        unset($this->data[$id]);

        return true;
    }

    /**
     * Clean some cache records
     *
     * Available modes are :
     * Zend_Cache::CLEANING_MODE_ALL (default)    => remove all cache entries ($tags is not used)
     * Zend_Cache::CLEANING_MODE_OLD              => remove too old cache entries ($tags is not used)
     * Zend_Cache::CLEANING_MODE_MATCHING_TAG     => remove cache entries matching all given tags
     *                                               ($tags can be an array of strings or a single string)
     * Zend_Cache::CLEANING_MODE_NOT_MATCHING_TAG => remove cache entries not {matching one of the given tags}
     *                                               ($tags can be an array of strings or a single string)
     * Zend_Cache::CLEANING_MODE_MATCHING_ANY_TAG => remove cache entries matching any given tags
     *                                               ($tags can be an array of strings or a single string)
     *
     * @param string $mode Clean mode
     * @param array $tags Array of tags
     * @return boolean true if no problem
     */
    public function clean($mode = Zend_Cache::CLEANING_MODE_ALL, $tags = array()) {
        /**
         * @var Array<string>|string $tags
         */
        switch($mode) {
        case Zend_Cache::CLEANING_MODE_ALL:
            $this->data = [];

            break;
        case Zend_Cache::CLEANING_MODE_OLD:
            // Nothing to do, our memory is per process and evicted after script

            break;
        case Zend_Cache::CLEANING_MODE_MATCHING_TAG:
        case Zend_Cache::CLEANING_MODE_NOT_MATCHING_TAG:
            $tags = is_array($tags) ? $tags : [$tags];

            foreach($this->data as $k => $v) {
                $matching = true;

                foreach($tags as $t) {
                    if( ! in_array($t, $v["tags"])) {
                        $matching = false;

                        break;
                    }
                }

                if($mode === Zend_Cache::CLEANING_MODE_MATCHING_TAG) {
                    if($matching) {
                        unset($this->data[$k]);
                    }
                }
                else {
                    if( ! $matching) {
                        unset($this->data[$k]);
                    }
                }
            }

            break;
        case Zend_Cache::CLEANING_MODE_MATCHING_ANY_TAG:
            $tags = is_array($tags) ? $tags : [$tags];

            foreach($this->data as $k => $v) {
                foreach($tags as $t) {
                    if(in_array($t, $v["tags"])) {
                        unset($this->data[$k]);

                        break;
                    }
                }
            }
        }

        return true;
    }

    /**
     * @param array $directives
     * @return void
     */
    public function setDirectives($directives) {

    }

    // Stuff we do not want
    public function getTmpDir() {
        throw new Exception(sprintf(
            "%s: Should be unused",
            __METHOD__
        ));
    }

    protected function _isGoodTmpDir($dir) {
        throw new Exception(sprintf(
            "%s: Should be unused",
            __METHOD__
        ));
    }
}
