<?php

declare(strict_types=1);

/**
 * Object wrapper for reading attributes using property access.
 *
 * @implements ArrayAccess<string, string>
 * @implements IteratorAggregate<string, string>
 */
final class Varien_Simplexml_Attributes implements ArrayAccess, Countable, IteratorAggregate {
    /**
     * @param Array<string, string> $data
     */
    final public function __construct(
        /**
         * @var Array<string, string>
         */
        private readonly array $data
    ) {}

    /**
     * Getter for attribute value.
     */
    public function __get(string $name): ?string {
        return $this->data[$name] ?? null;
    }

    public function __isset(string $name): bool {
        return isset($this->data[$name]);
    }

    /**
     * Attempts to set a property will throw since attributes are read-only.
     */
    public function __set(string $name, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set property %s::%s", __CLASS__, $name));
    }

    /**
     * Attempts to unset a property will throw since attributes are read-only.
     */
    public function __unset(string $name): void {
        throw new RuntimeException(sprintf("Attempting to unset property %s::%s", __CLASS__, $name));
    }

    public function count(): int {
        return count($this->data);
    }

    /**
     * @return Traversable<string, string>
     */
    public function getIterator(): Traversable {
        return new ArrayIterator($this->data);
    }

    /**
     * @param string $offset
     */
    public function offsetExists(mixed $offset): bool {
        return isset($this->data[$offset]);
    }

    /**
     * Integers index nodes, strings index attributes of the first node.
     *
     * @param string $offset
     * @return ?string
     */
    public function offsetGet(mixed $offset): mixed {
        return $this->data[$offset] ?? null;
    }

    /**
     * Attempts to set an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetSet(mixed $offset, mixed $value): void {
        throw new RuntimeException(sprintf("Attempting to set node attribute %s::%s", __CLASS__, (string)$offset));
    }

    /**
     * Attempts to unset an attribute will throw, modify the configuration
     * structure through Varien_Simplexml_Config instead.
     */
    public function offsetUnset(mixed $offset): void {
        throw new RuntimeException(sprintf("Attempting to unset node attribute %s::%s", __CLASS__, $offset));
    }
}
