<?php
/**
 * OpenMage
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available at https://opensource.org/license/osl-3-0-php
 *
 * @category   Mage
 * @package    Mage_Sales
 * @copyright  Copyright (c) 2006-2020 Magento, Inc. (https://www.magento.com)
 * @copyright  Copyright (c) 2020-2023 The OpenMage Contributors (https://www.openmage.org)
 * @license    https://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

/**
 * Order configuration model
 *
 * @category   Mage
 * @package    Mage_Sales
 *
 * @psalm-type States array{all:list<string>, visible:list<string>, invisible:list<string>, statuses:Array<string, list<string>>}
 */
class Mage_Sales_Model_Order_Config extends Mage_Core_Model_Config_Base
{
    /**
     * Statuses per state array
     *
     * @var Array<array>
     */
    protected $_stateStatuses = [];

    /**
     * Statuses array
     *
     * @var array
     */
    protected $_statuses;

    /**
     * States array
     *
     * @var ?States
     */
    private $_states;

    public function __construct()
    {
        $data = Mage::getConfig()->getNode('global/sales/order');

        assert($data instanceof Varien_Simplexml_Element);

        parent::__construct($data);
    }

    /**
     * @param string $status
     * @return ?Varien_Simplexml_Element
     */
    protected function _getStatus($status)
    {
        return $this->getNode('statuses/' . $status);
    }

    /**
     * @param string $state
     * @return ?Varien_Simplexml_Element
     */
    protected function _getState($state)
    {
        return $this->getNode('states/' . $state);
    }

    /**
     * Retrieve default status for state
     *
     * @param   string $state
     * @return  string|false
     */
    public function getStateDefaultStatus($state)
    {
        $status = false;
        $stateNode = $this->_getState($state);
        if ($stateNode) {
            $status = Mage::getModel('sales/order_status')
                ->loadDefaultByState($state);
            $status = $status->getStatus();
        }
        return $status;
    }

    /**
     * Retrieve status label
     *
     * @param   string $code
     * @return  string
     */
    public function getStatusLabel($code)
    {
        $key = $code . '/' . Mage::app()->getStore()->getStoreId();
        if (!isset($this->_statuses[$key])) {
            $status = Mage::getModel('sales/order_status')->load($code);
            $this->_statuses[$key] = $status->getStoreLabel();
        }
        return $this->_statuses[$key];
    }

    /**
     * State label getter
     *
     * @param   string $state
     * @return  string
     */
    public function getStateLabel($state)
    {
        $stateNode = $this->_getState($state);
        if ($stateNode) {
            $state = (string) $stateNode->label;
            return Mage::helper('sales')->__($state);
        }
        return $state;
    }

    /**
     * Retrieve all statuses
     *
     * @return array
     */
    public function getStatuses()
    {
        return Mage::getResourceModel('sales/order_status_collection')
            ->toOptionHash();
    }

    /**
     * Order states getter
     *
     * @return array
     */
    public function getStates()
    {
        $states = [];
        foreach ($this->getNode('states')?->children() ?? [] as $state) {
            $label = (string) $state->label;
            $states[$state->getName()] = Mage::helper('sales')->__($label);
        }
        return $states;
    }

    /**
     * Retrieve statuses available for state
     * Get all possible statuses, or for specified state, or specified states array
     * Add labels by default. Return plain array of statuses, if no labels.
     *
     * @param string|Array<string> $state
     * @param bool $addLabels
     * @return array
     */
    public function getStateStatuses($state, $addLabels = true)
    {
        if (is_array($state)) {
            $key = implode("|", $state) . $addLabels;
        } else {
            $key = $state . $addLabels;
        }
        if (isset($this->_stateStatuses[$key])) {
            return $this->_stateStatuses[$key];
        }
        $statuses = [];
        if (empty($state) || !is_array($state)) {
            $state = [$state];
        }
        foreach ($state as $_state) {
            $stateNode = $this->_getState($_state);
            if ($stateNode) {
                $collection = Mage::getResourceModel('sales/order_status_collection')
                    ->addStateFilter($_state)
                    ->orderByLabel();
                /** @var Mage_Sales_Model_Order_Status $status */
                foreach ($collection as $status) {
                    $code = $status->getStatus();
                    if ($addLabels) {
                        $statuses[$code] = $status->getStoreLabel();
                    } else {
                        $statuses[] = $code;
                    }
                }
            }
        }
        $this->_stateStatuses[$key] = $statuses;
        return $statuses;
    }

    /**
     * Retrieve state available for status
     * Get all assigned states for specified status
     *
     * @param string $status
     * @return array
     */
    public function getStatusStates($status)
    {
        $states = [];
        $collection = Mage::getResourceModel('sales/order_status_collection')->addStatusFilter($status);
        foreach ($collection as $state) {
            $states[] = $state;
        }
        return $states;
    }

    /**
     * Retrieve states which are visible on front end
     *
     * @return list<string>
     */
    public function getVisibleOnFrontStates()
    {
        return $this->_getStates()['visible'];
    }

    /**
     * Get order states, visible on frontend
     *
     * @return list<string>
     */
    public function getInvisibleOnFrontStates()
    {
        return $this->_getStates()['invisible'];
    }

    /**
     * If not yet initialized, loads the "_states" array object.
     *
     * @return States
     */
    private function _getStates(): array
    {
        if (null === $this->_states) {
            $this->_states = array(
                'all'       => [],
                'visible'   => [],
                'invisible' => [],
                'statuses'  => [],
            );
            foreach ($this->getNode('states')?->children() ?? [] as $state) {
                $name = $state->getName();
                $this->_states['all'][] = $name;
                $isVisibleOnFront = (string)$state->visible_on_front;
                if ((bool)$isVisibleOnFront || ($state->visible_on_front && $isVisibleOnFront == '')) {
                    $this->_states['visible'][] = $name;
                } else {
                    $this->_states['invisible'][] = $name;
                }
                foreach ($state->statuses->children() as $statusName => $_) {
                    $this->_states['statuses'][$name][] = $statusName;
                }
            }
        }

        return $this->_states;
    }
}
