<?php
/**
 * OpenMage
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available at https://opensource.org/license/osl-3-0-php
 *
 * @category   Mage
 * @package    Mage_Index
 * @copyright  Copyright (c) 2006-2020 Magento, Inc. (https://www.magento.com)
 * @copyright  Copyright (c) 2019-2023 The OpenMage Contributors (https://www.openmage.org)
 * @license    https://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

/**
 * Lock model
 *
 * @category   Mage
 * @package    Mage_Index
 */
class Mage_Index_Model_Lock
{
    /**
     * Lock storage config path
     */
    public const STORAGE_CONFIG_PATH = 'global/index/lock/storage';

    /**
     * Storage instance
     *
     * @var Mage_Index_Model_Lock_Storage_Interface
     */
    protected $_storage;

    /**
     * Singleton instance
     *
     * @var Mage_Index_Model_Lock
     */
    protected static $_instance;

    /**
     * Array of registered DB locks
     *
     * @var array
     */
    protected static $_lockDb = [];

    protected function __construct()
    {
        register_shutdown_function([$this, 'shutdownReleaseLocks']);
    }

    /**
     * Get lock singleton instance
     *
     * @return $this
     */
    public static function getInstance()
    {
        if (!self::$_instance instanceof self) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Release all locks on application shutdown
     */
    public function shutdownReleaseLocks()
    {
        // TODO: Do we really need to release the locks? They are autoreleased
        // when we close the connection.
        foreach (self::$_lockDb as $lockDb) {
            $this->releaseLock($lockDb);
        }
    }

    /**
     * Set named lock
     *
     * @param string $lockName
     * @param bool $file
     * @param bool $block
     * @return bool
     */
    public function setLock($lockName, $file = false, $block = false)
    {
        if ($this->_getLockStorage()->setLock($lockName)) {
            self::$_lockDb[$lockName] = $lockName;
            return true;
        }
        return false;
    }

    /**
     * Release named lock by name
     *
     * @param string $lockName
     * @param bool $file
     * @return bool
     */
    public function releaseLock($lockName, $file = false)
    {
        if ($this->_getLockStorage()->releaseLock($lockName)) {
            unset(self::$_lockDb[$lockName]);
            return true;
        }
        return false;
    }

    /**
     * Check whether the named lock exists
     *
     * @param string $lockName
     * @param bool $file
     * @return bool
     */
    public function isLockExists($lockName, $file = false)
    {
        return (bool) $this->_getLockStorage()->isLockExists($lockName);
    }

    /**
     * Get lock storage model
     *
     * @return Mage_Index_Model_Lock_Storage_Interface
     */
    protected function _getLockStorage()
    {
        if (!$this->_storage instanceof Mage_Index_Model_Lock_Storage_Interface) {
            $config = Mage::getConfig()->getNode(self::STORAGE_CONFIG_PATH);
            $this->_storage = Mage::getModel($config->model);
        }
        return $this->_storage;
    }
}
