<?php

use Awardit\MagentoPsr\Psr3\Logger;
use Awardit\SimpleEvent\Aws\SqsEventListener;
use Awardit\SimpleEvent\{
    EventHandlerInterface,
    EventInterface,
    EventRegistry,
    Metadata,
};
use Awardit\SimpleEvent\Event\{
    ErpProduct,
    ProductPricingPrice,
};
use Aws\Credentials\Credentials;
use Aws\Sqs\SqsClient;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Worker implements EventHandlerInterface
{
    private Awardit_EventListener_Handler_ErpProduct $erpProductHandler;
    private Awardit_EventListener_Handler_ProductPricingPrice $productPricingPriceHandler;
    private LoggerInterface $logger;

    public function __construct()
    {
        $this->logger = new Logger('event-listener');
        $this->erpProductHandler = new Awardit_EventListener_Handler_ErpProduct($this->logger);
        $this->productPricingPriceHandler = new Awardit_EventListener_Handler_ProductPricingPrice($this->logger);
    }

    /**
     * Entry point method for listener.
     */
    public function run(): void
    {
        $credentials = new Credentials(
            Mage::getStoreConfig('awardit_aws/credentials/access_key_id') ?: '',
            Mage::getStoreConfig('awardit_aws/credentials/secret_access_key') ?: '',
            Mage::getStoreConfig('awardit_aws/credentials/session_token') ?: null,
        );
        $config = [
            'credentials'   => $credentials,
            'region'        => Mage::getStoreConfig('awardit_aws/core/region') ?: null,
        ];
        $queueUrl =  Mage::getStoreConfig('awardit_aws/sqs/queue_url');

        $registry = new EventRegistry([
            ErpProduct::class,
            ProductPricingPrice::class,
        ]);
        $sqsClient = new SqsClient($config);
        $listener = new SqsEventListener($this->logger, $sqsClient, $queueUrl, $registry, 10);
        $listener->run($this);
    }

    /**
     * Delegator for incoming events.
     * Return true if event is fully handled, false to return it to queue.
     * @psalm-suppress InternalMethod
     */
    public function handle(EventInterface $event, Metadata $meta): bool
    {
        try {
            $this->logger->debug('Received event: {type}', [
                'type' => $event->getMessageType(),
                'entity' => $event->getEntityId(),
                'payload' => $event->formatMessagePayload(),
                'deleted' => $event->getIsDeleted(),
                'correlationId' => $meta->getCorrelationId(),
            ]);
            switch (get_class($event)) {
                case ErpProduct::class:
                    return $this->erpProductHandler->handle($event, $meta);
                case ProductPricingPrice::class:
                    return $this->productPricingPriceHandler->handle($event, $meta);
                default:
                    // No-op, consume event
                    return true;
            }
        } catch (Awardit_EventListener_EventException $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
                'correlationId' => $meta->getCorrelationId(),
            ]);
            return $e->isPermanent();
        } catch (Zend_Db_Statement_Exception $e) {
            // SQLSTATE[HY000]: General error: 2006 MySQL server has gone away
            //
            // We match a subset in case the error message changes from the driver.
            if (stripos($e->getMessage(), "2006 MySQL server has gone away") !== false) {
                // We now have a poisoned database connector, need to completely restart.
                $this->logger->error("Stopping process due to failed database connection: " . $e->getMessage(), [
                    'exception' => $e,
                    'correlationId' => $meta->getCorrelationId(),
                ]);

                exit(-1);
            } else {
                // Still want to indicate that we failed, to retry again
                throw $e;
            }
        } finally {
            // Collect garbage
            gc_collect_cycles();
        }
    }
}
