<?php

use Awardit\MagentoPsr\Psr3\Logger;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Model_Observer
{
    private Awardit_EventListener_Helper_Data $helper;
    private LoggerInterface $logger;

    public function __construct()
    {
        $this->helper = new Awardit_EventListener_Helper_Data();
        $this->logger = new Logger('event-observer');
    }

    // Exposed method called by product-save event
    public function beforeProductSave(Varien_Event_Observer $observer): void
    {
        $product = $observer->getProduct();
        if (empty($product) || !$this->shouldHandle($product)) {
            return; // Should not be handled by this observer
        }
        if ($product->getStoreId()) {
            $this->updateLocal($product);
        } else {
            $this->updateGlobal($product);
        }
    }

    public function afterProductSave(Varien_Event_Observer $observer): void
    {
        $product = $observer->getProduct();
        if (empty($product) || !empty($product->getStoreId()) || !$this->shouldHandle($product)) {
            return; // Should not be handled by this observer
        }
        $this->evalLocals($product);
    }

    // Update global product (store=0)
    private function updateGlobal(Mage_Catalog_Model_Product $global): void
    {
        if (!empty($global->getStoreId())) {
            return; // Not global
        }
        $erpData = $this->getErpData($global);
        if (is_null($erpData)) {
            return; // Not created from event
        }

        $enabled = !$erpData['archived'] && $global->getEnriched();
        if ($enabled) {
            $global->setStatus(Mage_Catalog_Model_Product_Status::STATUS_ENABLED);
        } else {
            $global->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
        }

        $visible = $enabled && $erpData['visible'];
        if ($visible) {
            $global->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH);
        } else {
            $global->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE);
        }

        $global->setForcedEvaluation(true);

        $this->logger->debug('Event observer global: {sku}', [
            'sku' => $global->getSku(),
            'enriched' => $global->getEnriched(),
            'erp.archived' => $erpData['archived'],
            'erp.visible' => $erpData['visible'],
            'enabled' => $enabled,
            'visible' => $visible,
        ]);
    }

    private function evalLocals(Mage_Catalog_Model_Product $global): void
    {
        if (!empty($global->getStoreId())) {
            return; // Not on global
        }
        foreach (Mage::app()->getStores(true) as $store) {
            if (empty($store->getId())) {
                continue; // Not on global
            }

            $local = $this->helper->loadProduct($global->getSku(), $store->getId());

            // Trigger this observer on local and save
            $local->setForcedEvaluation(true);
            $local->save();
        }
    }

    // Update local product (store=x)
    private function updateLocal(Mage_Catalog_Model_Product $local): void
    {
        if (empty($local->getStoreId())) {
            return; // Not local
        }
        $erpData = $this->getErpData($local);
        if (is_null($erpData)) {
            return; // Not created from event
        }
        $priceData = $this->getPriceData($local);
        if (is_null($priceData)) {
            return; // Not created from event
        }
        $salePrice = $this->getSalePrice($local, $priceData);

        $enabled = !$erpData['archived'] && !$priceData['archived'] && $local->getEnriched();
        if ($enabled) {
            $local->setStatus(Mage_Catalog_Model_Product_Status::STATUS_ENABLED);
        } else {
            $local->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
        }

        $visible = $enabled && $erpData['visible'] && !is_null($salePrice);
        if ($visible) {
            $local->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH);
        } else {
            $local->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE);
        }

        $this->logger->debug('Event observer local: {sku} for {store}', [
            'sku' => $local->getSku(),
            'store' => $local->getStore()->getCode(),
            'enriched' => $local->getEnriched(),
            'erp.archived' => $erpData['archived'],
            'erp.visible' => $erpData['visible'],
            'price.archived' => $priceData['archived'],
            'enabled' => $enabled,
            'visible' => $visible,
        ]);
    }

    private function getSalePrice(Mage_Catalog_Model_Product $local, array $priceData): ?float
    {
        foreach ($priceData['prices'] as $price) {
            if ($price['type'] == 'sale' && $price['currency'] == $local->getStore()->getBaseCurrencyCode()) {
                return $price['price'] === '' ? null : floatval($price['price']);
            }
        }
        return null;
    }

    private function shouldHandle(Mage_Catalog_Model_Product $product): bool
    {
        if ($product->getForcedEvaluation()) {
            return true;
        }
        foreach (
            [
                'status',
                'visibility',
                'enriched',
                'price',
                'msrp',
                'invoice_price',
                'purchase_price',
            ] as $field
        ) {
            if ($product->dataHasChangedFor($field)) {
                return true;
            }
        }
        return false;
    }

    private function getErpData(Mage_Catalog_Model_Product $product): ?array
    {
        $product_id = $product->getId();
        if ($product_id === null) {
            return null;
        }
        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpLog->loadLatest($product_id, null, 'ErpProduct');
        if ($erpLog->getId() === null) {
            return null;
        }
        return array_merge([
            'archived' => false,
            'visible' => false,
        ], $erpLog->getEventData());
    }

    private function getPriceData(Mage_Catalog_Model_Product $product): ?array
    {
        $product_id = $product->getId();
        if ($product_id === null) {
            return null;
        }
        $priceLog = Mage::getModel('awardit_eventlistener/event_log');
        $priceLog->loadLatest($product_id, $product->getStoreId(), 'ProductPricingPrice');
        if ($priceLog->getId() === null) {
            return null;
        }
        return array_merge([
            'archived' => false,
            'visible' => false,
        ], $priceLog->getEventData());
    }
}
