<?php

declare(strict_types=1);

use Awardit\MagentoPsr\Psr3\Logger;
use Awardit\SimpleEvent\Event\ErpProduct;
use Awardit\SimpleEvent\Event\ProductPricingPrice;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Awardit\SimpleEvent\Metadata;
use Aws\Exception\CredentialsException;
use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

class EventTest extends TestCase
{
    private static string $sku;
    private static int $store_id = 2;
    private static string $store_code = 'testing';
    private Awardit_EventListener_Helper_Data $helper;

    public static function setUpBeforeClass(): void
    {
        // Need the same SKU in all tests.
        self::$sku = 'test-' . rand(1000, 9999);
        /** @var Mage_Core_Model_Config $config */
        $config = Mage::getConfig();
        $config->saveConfig('catalog/price/scope', '1');
        $config->saveConfig(
            'awardit_aws/mappings/tax_mapping',
            'a:1:{s:18:"_1729170243290_290";a:2:{s:12:"erp_tax_code";s:11:"visma_se_25";s:12:"tax_class_id";s:1:"2";}}'
        );
    }

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init(self::$store_code);
        Mage::app()->setCurrentStore('0');
        $this->helper = Mage::helper('awardit_eventlistener');
    }

    public function tearDown(): void
    {
        MagentoManager::logQueries();
    }

    public function testErpCreate(): void
    {
        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Create global product
        $erpEvent = $this->createErpProduct([
            'sku'   => self::$sku,
            'name'  => 'Test product 1',
        ]);
        $erpHandler->handle($erpEvent, new Metadata('test-correlation-id', 'test-sender'));

        // Load and check global product
        $global = $this->loadProduct(self::$sku, 0);
        $this->assertEquals(self::$sku, $global->getSku());
        $this->assertEquals(0, $global->getStoreId());
        $this->assertEquals('Test product 1', $global->getName());
        $this->assertEquals('Test product 1', $global->getDescription());
        $this->assertEquals('Test product 1', $global->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());
        $this->assertFalse($global->isAvailable());
        $this->assertFalse($global->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $global->getTypeId());
        $this->assertEquals(0, $global->getPrice());
        $this->assertEquals(0, $global->getMsrp());
        $this->assertNull($global->getWeight());
        $this->assertEquals('Taxable Goods', $global->getAttributeText('tax_class_id'));
        $this->assertFalse($global->getAttributeText('manufacturer'));
        $this->assertFalse($global->getAttributeText('supplier'));
        $this->assertNull($global->getPurchasePrice());
        $this->assertNull($global->getInvoicePrice());

        // Load and check local product
        $local = $this->loadProduct(self::$sku, self::$store_id);
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1', $local->getName());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $local->getVisibility());
        $this->assertFalse($local->isAvailable());
        $this->assertFalse($local->isSalable());

        // Load and check global event log
        $erpLog->loadLatest($global->getEntityId(), null, 'ErpProduct');
        $this->assertEquals(1, $erpLog->getRevision());
        $this->assertEquals($erpEvent->formatMessagePayload(), $erpLog->getEventData());
    }

    public function testPriceCreate(): void
    {
        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingHandler = new Awardit_EventListener_Handler_ProductPricingPrice(new Logger('test'));

        // Create local product
        $pricingEvent = $this->createProductPricingPrice([
            'sku'       => self::$sku,
            'priceList' => self::$store_code,
            'prices' => [
                [
                    'type'  => 'sale',
                    'price' => 123.45,
                    'lowest' => 112.19,
                ],
                [
                    'type'  => 'recommended',
                    'price' => 245.67,
                ],
                [
                    'type'  => 'invoice',
                    'price' => 345.67,
                ],
                [
                    'type'  => 'purchase',
                    'price' => 99.99,
                ],
            ],
            'revision'  => 1,
        ]);
        $pricingHandler->handle($pricingEvent, new Metadata('test-correlation-id', 'test-sender'));

        // Load and check global product
        $global = $this->loadProduct(self::$sku, 0);
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());

        // Load and check local product
        $local = $this->loadProduct(self::$sku, self::$store_id);
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1', $local->getName());
        $this->assertEquals('Test product 1', $local->getDescription());
        $this->assertEquals('Test product 1', $local->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $local->getVisibility());
        $this->assertFalse($local->isAvailable());
        $this->assertFalse($local->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $local->getTypeId());
        $this->assertNull($local->getWeight());
        $this->assertEquals('Taxable Goods', $local->getAttributeText('tax_class_id'));
        $this->assertFalse($local->getAttributeText('manufacturer'));
        $this->assertFalse($global->getAttributeText('supplier'));
        $this->assertEquals(123.45, $local->getPrice());
        $this->assertEquals(245.67, $local->getMsrp());
        $this->assertEquals(99.99, $local->getPurchasePrice());
        $this->assertEquals(345.67, $local->getInvoicePrice());
        $this->assertEquals(112.19, $local->getLowestSalePrice());

        // Load and check local event log
        $pricingLog->loadLatest($local->getEntityId(), self::$store_id, 'ProductPricingPrice');
        $this->assertEquals(1, $pricingLog->getRevision());
        $this->assertEquals($pricingEvent->formatMessagePayload(), $pricingLog->getEventData());
    }

    public function testErpUpdate(): void
    {
        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Update global product
        $erpEvent = $this->createErpProduct([
            'sku'               => self::$sku,
            'name'              => 'Test product 1 changed',
            'weight'            => 123.45,
            'sellable'          => true,
            'visible'           => true,
            'brand'             => 'Test brand',
            'supplier'          => 'Test supplier',
            'supplierSku'       => 'TS01',
            'taxClass'          => 'visma_se_25',
            'allowShortPick'    => true,
            'msrp'              => [
                [
                    'currency'  => 'SEK',
                    'price'     => '98.76',
                ],
                [
                    'currency'  => 'EUR',
                    'price'     => '9.87',
                ],
            ],
            'purchasePrice'     => [
                'price'     => '250',
                'currency'  => 'SEK'
            ],
            'revision'          => 2,
        ]);
        $erpHandler->handle($erpEvent, new Metadata('test-correlation-id', 'test-sender'));

        // Load and check global product
        $global = $this->loadProduct(self::$sku, 0);
        $this->assertEquals(self::$sku, $global->getSku());
        $this->assertEquals(0, $global->getStoreId());
        $this->assertEquals('Test product 1', $global->getName());
        $this->assertEquals('Test product 1', $global->getDescription());
        $this->assertEquals('Test product 1', $global->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());
        $this->assertFalse($global->isAvailable());
        $this->assertFalse($global->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $global->getTypeId());
        $this->assertEquals(0, $global->getPrice());
        $this->assertEquals(98.76, $global->getMsrp());
        $this->assertEquals(123.45, $global->getWeight());
        $this->assertEquals('Taxable Goods', $global->getAttributeText('tax_class_id'));
        $this->assertEquals('Test brand', $global->getAttributeText('manufacturer'));
        $this->assertEquals('Test supplier', $global->getAttributeText('supplier'));
        $this->assertEquals(250, $global->getPurchasePrice());

        // Load and check local product
        $local = $this->loadProduct(self::$sku, self::$store_id);
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1', $local->getName());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $local->getVisibility());
        $this->assertFalse($local->isAvailable());
        $this->assertFalse($local->isSalable());

        // Load and check global event log
        $erpLog->loadLatest($global->getEntityId(), null, 'ErpProduct');
        $this->assertEquals(2, $erpLog->getRevision());
        $this->assertEquals($erpEvent->formatMessagePayload(), $erpLog->getEventData());
    }

    public function testEnableUpdate(): void
    {

        $local = $this->helper->loadProduct(self::$sku, self::$store_id);
        $local->setEnriched(1);
        $local->save();

        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingHandler = new Awardit_EventListener_Handler_ProductPricingPrice(new Logger('test'));

        // Update local product
        $pricingEvent = $this->createProductPricingPrice([
            'sku'       => self::$sku,
            'priceList' => self::$store_code,
            'prices' => [
                [
                    'type'  => 'sale',
                    'price' => 234.67,
                ],
            ],
            'revision'  => 2,
        ]);
        $pricingHandler->handle($pricingEvent, new Metadata('test-correlation-id', 'test-sender'));

        // Load and check local product
        $local = $this->loadProduct(self::$sku, self::$store_id);
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1', $local->getName());
        $this->assertEquals('Test product 1', $local->getDescription());
        $this->assertEquals('Test product 1', $local->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_ENABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH, $local->getVisibility());
        $this->assertTrue($local->isAvailable());
        $this->assertTrue($local->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $local->getTypeId());
        $this->assertEquals(234.67, $local->getPrice());
        $this->assertEquals(245.67, $local->getMsrp());
        $this->assertEquals(234.67, $local->getLowestSalePrice());
        $this->assertEquals(123.45, $local->getWeight());
        $this->assertEquals('Test brand', $local->getAttributeText('manufacturer'));
        $this->assertEquals('Test supplier', $local->getAttributeText('supplier'));

        // Load and check local event log
        $pricingLog->loadLatest($local->getEntityId(), self::$store_id, 'ProductPricingPrice');
        $this->assertEquals(2, $pricingLog->getRevision());
        $this->assertEquals($pricingEvent->formatMessagePayload(), $pricingLog->getEventData());
    }

    public function testErpArchive(): void
    {
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Update global product
        $erpEvent = $this->createErpProduct([
            'sku'               => self::$sku,
            'name'              => 'Test product archived',
            'archived'          => true,
            'revision'          => 3,
        ]);
        $erpHandler->handle($erpEvent, new Metadata('test-correlation-id', 'test-sender'));

        // Load and check global product - should have been removed
        $global = $this->loadProduct(self::$sku, 0);
        $this->assertNull($global->getId());
        $this->assertNull($global->getSku());

        // Load and check local product - should have been removed
        $local = $this->loadProduct(self::$sku, self::$store_id);
        $this->assertNull($local->getId());
        $this->assertNull($local->getSku());
    }

    public function testWorker(): void
    {
        // Just test that local code runs, i.e. expect SQS event read to fail
        $worker = new Awardit_EventListener_Worker();
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Missing required client configuration options');
        $worker->run();
    }

    private function loadProduct(string $sku, int $store): Mage_Catalog_Model_Product
    {
        $product = Mage::getModel('catalog/product');
        $product->setStoreId($store);
        $product->load($product->getIdBySku($sku));
        return $product;
    }

    /**
     * @psalm-suppress InvalidArgument
     */
    private function createErpProduct(array $input): ErpProduct
    {
        $data = array_merge([
            'sku'               => 'GEN-PROD',
            'name'              => 'Generated product',
            'type'              => 'physical',
            'weight'            => null,
            'archived'          => false,
            'sellable'          => false,
            'visible'           => false,
            'allowShortPick'    => false,
            'manufacturer'      => null,
            'mpn'               => null,
            'taxClass'          => 'visma_se_25',
            'msrp'              => [],
            'purchasePrice'     => [],
            'revision'          => 1,
            'categories'        => [],
        ], $input);
        // TODO: Use rest-spread once we can use named parameters
        $instance = new ErpProduct($data);
        $instance->setRevision($data['revision']);
        return $instance;
    }

    /**
     * @psalm-suppress InvalidArgument
     */
    private function createProductPricingPrice(array $input): ProductPricingPrice
    {
        $data = array_merge([
            'sku'               => 'GEN-PROD',
            'priceList'         => 'GEN-STORE',
            'revision'          => 1,
            'archived'          => false,
            'prices'            => [],
            'campaign'          => null,
        ], $input);
        $data['prices'] = array_values(array_map(function (array $item) {
            return new Price([
                'type' => $item['type'],
                'currency' => 'SEK',
                'price' => (string)$item['price'],
                'lowestPrice30days' => (string)($item['lowest'] ?? $item['price']),
                'customerGroup' => null,
            ]);
        }, $input['prices']));
        // TODO: Use rest-spread once we can use named parameters
        $instance = new ProductPricingPrice($data);
        return $instance;
    }
}
