<?php

use Awardit\SimpleEvent\Event\ErpProduct;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Handler_ErpProduct
{
    private LoggerInterface $logger;
    private Awardit_EventListener_Helper_Data $helper;

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
        $this->helper = Mage::helper('awardit_eventlistener');
    }

    /**
     * Handler for ErpProduct events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(ErpProduct $erpProduct): bool
    {
        $currency = Mage::app()->getBaseCurrencyCode();
        $lastLog = Mage::getModel('awardit_eventlistener/event_log');
        $product = Mage::getModel('catalog/product');
        $product->setStoreId(0);
        $id = $product->getIdBySku($erpProduct->getEntityId());
        if ($id) {
            // Existing entry
            $product->load($id);
            $lastLog->loadLatest($product->getEntityId(), null, 'ErpProduct');
            if ($lastLog->getRevision() >= $erpProduct->getRevision()) {
                $this->logger->debug(
                    "Ignore product '{$product->getSku()}' ({$product->getEntityId()}), old revision",
                    [
                        "revision" => [
                            "last" => $lastLog->getRevision(),
                            "current" => $erpProduct->getRevision(),
                        ],
                        "product" => [
                            "entityId" => $product->getEntityId(),
                            "sku" => $product->getSku(),
                        ],
                    ]
                );
                return true; // Ignore, revision already handled
            }
            if ($erpProduct->getIsDeleted()) {
                $this->logger->debug("Remove product '{$product->getSku()}' ({$product->getEntityId()}), is deleted", [
                    "product" => [
                        "entityId" => $product->getEntityId(),
                        "sku" => $product->getSku(),
                    ],
                ]);
                $product->delete();
                return true;
            }
        } else {
            // New entry
            if ($erpProduct->getIsDeleted()) {
                $this->logger->debug("Ignore product '{$erpProduct->getEntityId()}', is deleted", [
                    "product" => [
                        "sku" => $erpProduct->getEntityId(),
                    ],
                ]);
                return true; // Ignore, no need to create
            }
            $product->setSku($erpProduct->getEntityId());
            $product->setName($erpProduct->name ?? '');
            $product->setDescription($erpProduct->name ?? '');
            $product->setShortDescription($erpProduct->name ?? '');
            $product->setAttributeSetId($product->getDefaultAttributeSetId());
            $product->setTypeId($this->getType($erpProduct->type));
            $product->setAwarditDropship($erpProduct->type == 'dropship');
            $product->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
            $product->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE);
            $product->setCategoryIds([]);
            $product->setPrice(0.0);
            $product->setEnriched(0);
            $product->save(); // Pre-save
        }

        // @todo: No property that correspons to "sellable" input
        $product->setStockData([
            'use_config_manage_stock' => 0,
            'manage_stock' => !$erpProduct->allowShortPick,
            'backorders' => (int)$erpProduct->allowShortPick, // @todo: Uncertain
        ]);
        $product->setWeight($erpProduct->weight);
        $product->setTaxClassId($this->helper->getTaxClassId($erpProduct->taxClass));
        // @deprecated manufacturer
        $manufacturer = $erpProduct->brand ?? $erpProduct->manufacturer ?? null;
        $product->setManufacturer($this->getOptionId('manufacturer', $manufacturer));
        $product->setSupplier($this->getOptionId('supplier', $erpProduct->supplier));
        foreach ($erpProduct->msrp as $price) {
            if ($currency == $price['currency']) {
                $product->setMsrp((float)$price['price']);
            }
        }
        if ($erpProduct->purchasePrice && $currency == $erpProduct->purchasePrice['currency']) {
            $product->setPurchasePrice((float)$erpProduct->purchasePrice['price']);
        }

        $newLog = Mage::getModel('awardit_eventlistener/event_log');
        $newLog->setPreviousLogId($lastLog->getLogId());
        $newLog->setRevision($erpProduct->getRevision());
        $newLog->setProductId($product->getEntityId());
        $newLog->setEventType('ErpProduct');
        $newLog->setEventData($erpProduct->formatMessagePayload());
        $newLog->save();

        $product->setForcedEvaluation(true);
        $product->save();

        $this->logger->info("Updated ERP product '{$product->getSku()}' ({$product->getEntityId()})", [
            "product" => [
                "entityId" => $product->getEntityId(),
                "sku" => $product->getSku(),
            ],
        ]);

        return true;
    }

    private function getType(string $type): string
    {
        switch ($type) {
            case 'dropship':
            case 'physical':
                return Mage_Catalog_Model_Product_Type::TYPE_SIMPLE;
            case 'virtual':
                return Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL;
            default:
                throw new RuntimeException("Product type '{$type}' not supported.");
        }
    }

    private function getOptionId(string $code, ?string $value): ?int
    {
        /** @psalm-suppress RiskyTruthyFalsyComparison */
        return Mage::helper('integration')->getOptionId($code, $value, !empty($value));
    }
}
