<?php

use Awardit\SimpleEvent\Event\ProductPricingPrice;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Handler_ProductPricingPrice
{
    private LoggerInterface $logger;
    private Awardit_EventListener_Helper_Data $helper;

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
        $this->helper = new Awardit_EventListener_Helper_Data();
    }

    /**
     * Handler for ProductPricingPrice events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(ProductPricingPrice $productPrices): bool
    {
        $store = $this->getStore($productPrices->priceList);
        $product = $this->getProduct($productPrices->getEntityId(), $store);
        $currency = $store->getBaseCurrencyCode();

        $product->setWebsiteIds(array_merge($product->getWebsiteIds(), [$store->getWebsiteId()]));
        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingLog->loadLatest($product->getEntityId(), $store->getStoreId(), 'ProductPricingPrice');
        if ($pricingLog->getRevision() >= $productPrices->getRevision()) {
            $this->logger->debug("Ignore product '{$product->getSku()}' ({$product->getEntityId()}), old revision", [
                "revision" => [
                    "last" => $pricingLog->getRevision(),
                    "current" => $productPrices->getRevision(),
                ],
                "product" => [
                    "entityId" => $product->getEntityId(),
                    "sku" => $product->getSku(),
                ],
            ]);
            return true; // Ignore, revision already handled
        }

        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpLog->loadLatest($product->getEntityId(), null, 'ErpProduct');

        // Monetary prices
        foreach ($productPrices->prices as $price) {
            if ($currency != $price->currency) {
                continue; // Ignore
            }
            $this->handlePrice($price, $product, $store);
        }

        // Point prices
        $helper = Mage::helper('points_core');
        $pointTypes = array_keys($helper->getTypeProviders());
        foreach ($productPrices->prices as $price) {
            if (!in_array($price->currency, $pointTypes)) {
                continue; // Ignore
            }
            $this->handlePoints($price, $product, $store);
        }
        $product->setForcedEvaluation(true);
        $product->save();

        $newLog = Mage::getModel('awardit_eventlistener/event_log');
        $newLog->setPreviousLogId($pricingLog->getLogId());
        $newLog->setRevision($productPrices->getRevision());
        $newLog->setProductId($product->getEntityId());
        $newLog->setStoreId($store->getStoreId());
        $newLog->setEventType('ProductPricingPrice');
        /** @psalm-suppress UndefinedDocblockClass */
        $newLog->setEventData($productPrices->formatMessagePayload());
        $newLog->save();

        $this->logger->info("Updated product price for '{$product->getSku()}' ({$product->getEntityId()})", [
            "product" => [
                "entityId" => $product->getEntityId(),
                "sku" => $product->getSku(),
            ],
            "storeCode" => $store->getCode(),
        ]);
        return true;
    }

    private function handlePrice(Price $price, Mage_Catalog_Model_Product $product, Mage_Core_Model_Store $store): void
    {
        /** @psalm-suppress TypeDoesNotContainNull */
        if (is_null($price->price)) {
            return; // Magento don't accept null, observer sets not visible
        }

        $amount = (float)$price->price;
        $lowest = (float)$price->lowestPrice30days;
        switch ($price->type) {
            case 'sale':
                $product->setPrice($amount);
                $product->setLowestSalePrice($lowest);
                break;
            case 'recommended':
                $product->setMsrp($amount);
                break;
            case 'invoice':
                $product->setInvoicePrice($amount);
                break;
            case 'purchase':
                $product->setPurchasePrice($amount);
                break;
            default:
                $this->logger->warning("Unsupported {$price->type} price for '{$product->getSku()}'", [
                    "product" => [
                        "entityId" => $product->getEntityId(),
                        "sku" => $product->getSku(),
                    ],
                    "price" => [
                        "type" => $price->type,
                        "amount" => $amount,
                        "currency" => $price->currency,
                    ],
                    "storeCode" => $store->getCode(),
                ]);
                return;
        }

        $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'", [
            "product" => [
                "entityId" => $product->getEntityId(),
                "sku" => $product->getSku(),
            ],
            "price" => [
                "type" => $price->type,
                "amount" => $amount,
                "currency" => $price->currency,
            ],
            "storeCode" => $store->getCode(),
        ]);
    }

    private function handlePoints(Price $price, Mage_Catalog_Model_Product $product, Mage_Core_Model_Store $store): void
    {
        /** @psalm-suppress TypeDoesNotContainNull */
        if (is_null($price->price)) {
            return; // Magento don't accept null, observer sets not visible
        }

        $pointPrice = Mage::getModel('points_core/product_price');
        $pointPrice->loadByStoreProductTypeCustomerGroupId($product->getStore(), $product, $price->currency);

        if ($pointPrice->getId() === null) {
            $pointPrice->setStore($product->getStore());
            $pointPrice->setProduct($product);
            $pointPrice->setType($price->currency);
        }
        $amount = (int)$price->price;
        switch ($price->type) {
            case 'sale':
                $pointPrice->setPrice($amount);
                $this->logger->info("Updated {$price->type} points for '{$product->getSku()}'", [
                    "product" => [
                        "entityId" => $product->getEntityId(),
                        "sku" => $product->getSku(),
                    ],
                    "price" => [
                        "type" => $price->type,
                        "amount" => $amount,
                        "currency" => $price->currency,
                    ],
                    "storeCode" => $store->getCode(),
                ]);
                break;
            case 'recommended':
            case 'invoice':
            case 'purchase':
                // @todo: Not default attribute
            default:
                $this->logger->warning("Unsupported {$price->type} points for '{$product->getSku()}'", [
                    "product" => [
                        "entityId" => $product->getEntityId(),
                        "sku" => $product->getSku(),
                    ],
                    "price" => [
                        "type" => $price->type,
                        "amount" => $amount,
                        "currency" => $price->currency,
                    ],
                    "storeCode" => $store->getCode(),
                ]);
                break;
        }
        $pointPrice->save();
    }

    private function getProduct(string $code, Mage_Core_Model_Store $store): Mage_Catalog_Model_Product
    {
        return $this->helper->loadProduct($code, $store->getId());
    }

    private function getStore(string $code): Mage_Core_Model_Store
    {
        try {
            $store = Mage::app()->getStore($code);
            if ($store === null) {
                 throw new Awardit_EventListener_EventException("Store '{$code}' do not exist", true);
            }
            return $store;
        } catch (Mage_Core_Model_Store_Exception $e) {
            throw new Awardit_EventListener_EventException("Store '{$code}' do not exist", true, $e);
        } catch (Throwable $t) {
            throw new Awardit_EventListener_EventException("Store '{$code}' error", false, $t);
        }
    }
}
